// swathprofile.cpp version 1.1 (Jan 2014)
// Author: Stefan Hergarten
// Tested on Linux 64 bit, compiled with g++(gcc) 4.7
// Execute without any command-line arguments to obtain a list of all supported arguments
// In case of any problems, suggestions for extensions, or just to obtain the latest
// version, please send an email to the author at stefan@hergarten.at. The author
// would also be glad to be notified about any useful applications.

#include <cstdlib>
#include <cstdio>
#include <string>
#include <cstring>
#include <climits>
#include <cfloat>
#include <cmath>
#include <vector> 

#define DEG *M_PI/180

#define DATATYPE float
#define NODATA -9999

using namespace std;

double scale = 1.; // defines the length of 1 unit in tracks in meters

class CNVal
{
  int     n;
  double  v, v2, min, max;

  public:

  CNVal()
  {
    n = 0;
    v = v2 = 0.;
    min = DBL_MAX;
    max = -DBL_MAX;
  }

  void flush()
  {
    v = v2 = n = 0;
    min = DBL_MAX;
    max = -DBL_MAX;
  }

  void operator += ( const double u )
  {
    ++n;
    v += u;
    v2 += u*u;
    if ( u > max )  max = u;
    if ( u < min )  min = u;
  }

  void operator += ( const CNVal u )
  {
    n += u.n;
    v += u.v;
    v2 += u.v2;
    if ( u.max > max )  max = u.max;
    if ( u.min < min )  min = u.min;
  }

  int getNumber()
  {
    return n;
  }

  double getValue()
  {
    return v;
  }

  double getMean()
  {
    return n ? v/n : 0.;
  }

  double getVar()
  {
    return n > 1 ? (v2-v*v/n)/(n-1) : 0.; 
  }

  double getMin()
  {
    return min; 
  }

  double getMax()
  {
    return max; 
  }
};  

char *readFile ( char *filename, int &len )
{
  FILE *fp = fopen ( filename, "rb" );
  fseek ( fp, 0, SEEK_END );
  len = ftell ( fp );
  fseek ( fp, 0, SEEK_SET );
  char *buf = (char*) malloc ( len+1 );
  fread( buf, len, 1, fp );
  buf[len] = '\0';
  fclose ( fp );
  return buf;
}

char *readFile ( char *filename )
{
  int  len;
  return readFile ( filename, len );
}

class Point
{
  public:
  double  lon, lat;

  Point operator * ( const double lambda )
  { 
    Point p;
    p.lon = lon*lambda;
    p.lat = lat*lambda;
    return p;
  }

  Point operator *= ( const double lambda )
  { 
    lon *= lambda;
    lat *= lambda;
    return *this;
  }

  Point operator + ( const Point p )
  { 
    Point q;
    q.lon = lon+p.lon;
    q.lat = lat+p.lat;
    return q;
  }

  Point operator += ( const Point p )
  { 
    lon += p.lon;
    lat += p.lat;
    return *this;
  }

  Point operator - ( const Point p )
  { 
    Point q;
    q.lon = lon-p.lon;
    q.lat = lat-p.lat;
    return q;
  }

  Point operator -= ( const Point p )
  { 
    lon -= p.lon;
    lat -= p.lat;
    return *this;
  }

  double length()
  {
    if ( scale < 1e5 )
      return sqrt(lon*lon+lat*lat);
    double tmp = lon*cos(lat DEG);
    return scale*sqrt(tmp*tmp+lat*lat);
  }	  

  double distance ( Point a )
  {
    a -= *this;
    return a.length();
  }

  double distance ( Point a, Point b )
  {
    b -= a;
    a -= *this;
    double f = 1.;
    if ( scale > 1e5 )
    {	    
      double c = cos(lat DEG); 
      a.lon *= c;
      b.lon *= c;
      f = scale;
    }
    return f*sqrt(fabs(a.lon*a.lon+a.lat*a.lat-pow((a.lon*b.lon+a.lat*b.lat),2)/(b.lon*b.lon+b.lat*b.lat)));
  }

  double distance ( Point a, Point b, Point &p, double &lambda )
  {
    b -= a;
    a -= *this;
    double c = 1., f = 1.;
    if ( scale > 1e5 )
    {	    
      c = cos(lat DEG); 
      a.lon *= c;
      b.lon *= c;
      f = scale;
    }
    double den = b.lon*b.lon+b.lat*b.lat;
    lambda = den ? -(a.lon*b.lon+a.lat*b.lat)/den : 0.;
    if ( lambda < 0. )  lambda = 0.;
    else 
      if ( lambda > 1. )  lambda = 1.;
    p = a + b*lambda;
    double d = f*sqrt(p.lon*p.lon+p.lat*p.lat);
    p.lon /= c;
    if ( a.lon*b.lat > a.lat*b.lon )  d = -d;
    p += *this;
    return d;
  }
};

class Track
{
  public:

  char           name[1024];
  vector<Point>  points;
  vector<double> len;
  Point          ll, ur;

  Track()
  {
    *name = '\0';
  }

  void computeLength()
  {
    len.push_back(0);
    for ( int i = 1; i < points.size(); ++i )
      len.push_back(len[i-1]+points[i].distance(points[i-1]));
//    for ( int i = 0; i < len.size(); ++i )
//      printf ( "%i %g\n", i, len[i] );
  }    

  void readGPX ( char *filename )
  {
    Point  c;
    char *p, *q, *r, *start, *end;
    char *buf = readFile ( filename );
    for ( p = buf; ( p = strstr(p,"<gpxx:rpt") ) != NULL; )
      memcpy ( p, "<trkpt   ", 9 );
    if ( ( start = strstr(buf,"<name") ) != NULL )
    {
      start = strchr(start,'>') + 1;
      while ( isspace(*start) )  ++start;
      end = strstr(start,"</name>") - 1;  
      while ( isspace(*end) )  --end;
      strncpy ( name, start, end+1-start );
      name[end+1-start] = '\0'; 
    }
    end = buf-1;
    while( ( start = strstr(end+1,"<trkpt") ) != NULL )
    {
      int ct = 1;
      for ( end = start+1; ct; ++end )
      {
        if ( *end == '<' && *(end+1) != '/' )  ++ct;
        else
          if ( *end == '/' )  --ct;
      }
      *end = '\0';
      c.lat = 0.; 
      if ( ( p = strstr(start,"lat") ) != NULL && p < end ) 
      {
        while ( !isdigit(*p) && *p != '-' )  ++p;
        c.lat = atof(p); 
      }
      c.lon = 0.;
      if ( ( p = strstr(start,"lon") ) != NULL && p < end ) 
      {
        while ( !isdigit(*p) && *p != '-' )  ++p;
        c.lon = atof(p); 
      }
      points.push_back(c);    
    }
  }

  void readKML ( char *filename )
  {
    Point  c;
    char     *p, *q, *coord;
    double   lon, lat;
    char *buf = readFile ( filename );
    strcpy ( name, filename );
    coord = strstr(buf,"<coordinates>") + 13;
    *(strstr(coord,"</coordinates>")+1) = '\0';
    p = coord;
    while ( 1 )
    {
      c.lon = strtod ( p, &q );
      if ( *q == '\0' || p == q )  break;
      p = q+1;
      c.lat = strtod ( p, &q );
      if ( *q == '\0' || p == q )  break;
      p = q+1;
      for ( q = p-1; q != '\0' && !isdigit(*q); ++q )
        if ( *q == ',' )    
        {
          strtod ( q+1, &p );
//          if ( *q == '\0' || p == q )  break;
          p = p+1;
          break;
        }
//printf ( "%e %e\n", c.lat, c.lon );
      points.push_back(c);    
    }
  } 

  void readCSV  ( char *filename )
  {
    Point  c;
    char *buf = readFile ( filename );
    for ( char *p = buf; *p != '\0'; ++p )
      if ( *p == ',' || *p == ';' )  *p = ' ';
    do
      if ( sscanf ( buf, "%le%le", &c.lon, &c.lat ) == 2 )  points.push_back(c);
    while ( ( buf = strchr(buf,'\n') )++ != NULL );
  }

  double read ( char *filename )
  {
    if ( strstr(filename,".gpx") != NULL || strstr(filename,".GPX") != NULL )
      readGPX ( filename );
    else
      if ( strstr(filename,".kml") != NULL || strstr(filename,".KML") != NULL )
        readKML ( filename );
      else
        readCSV ( filename );
//    if ( points.size() > 65535 )
//    {
//      printf ( "Error: Track contains more than 65535 points.\n" );
//      exit ( -1 );
//    }  
    if ( points.size() )
      printf ( "Read track from file %s with %i points.\n", filename, points.size() );  
//    computeLength();
  }

  void computeBBox( int istart = 0, int iend = 65535 )
  {
    ll.lon = DBL_MAX;
    ll.lat = DBL_MAX;
    ur.lon = -DBL_MAX;
    ur.lat = -DBL_MAX;
    if ( iend >= points.size() )  iend = points.size()-1;
    for ( int i = istart; i <= iend; ++i )
    {
      if ( points[i].lon < ll.lon )  ll.lon = points[i].lon;
      if ( points[i].lat < ll.lat )  ll.lat = points[i].lat;
      if ( points[i].lon > ur.lon )  ur.lon = points[i].lon;
      if ( points[i].lat > ur.lat )  ur.lat = points[i].lat;
    }
  }

  Track shorten ( double d, double lmax )
  {
    Track  s;
    strcpy ( s.name, name );
    s.points.push_back(points[0]);
    int last = 0;
    for ( int i = last+2; i < points.size(); ++i ) 
      for ( int j = last+1; j < i; ++j )
        if ( points[i].distance(points[last]) > lmax 
		|| points[j].distance(points[last],points[i]) > d )
        {
          last = i-1;
          s.points.push_back(points[last]);
          i = last+2;
          break;
        }     
    s.points.push_back(points.back());
    return s;
  }

  double distance ( Point x, Point &p, int imin = 0, int imax = 65536 )
  {
    int    ibest;	  
    double dist, lambda, lambdabest, mindist = DBL_MAX;
    Point a;
    if ( imin < 0 )  imin = 0;
    if ( imax >= points.size() )  imax = points.size()-1;
    for ( int i = imin+1; i <= imax; ++i )
    {
      double d = x.distance ( points[i-1], points[i], a, lambda );
      if ( fabs(d) < fabs(mindist) )
      {
	ibest = i;      
        mindist = d;
        lambdabest = lambda;
        p = a;
      }
    }
    if ( ibest == 1 && lambdabest == 0. 
         || ibest == points.size()-1 && lambdabest == 1. )  mindist = DBL_MAX;
    return mindist;
  }

  int findNearestPoint ( Point x, int imin, int imax )
  {
    int     ibest = 0;
    double  mindist = DBL_MAX;

    if ( imin < 0 )  imin = 0;
    if ( imax >= points.size() )  imax = points.size()-1;
    for ( int i = imin; i <= imax; ++i )
    {
      double d = x.distance ( points[i] );
      if ( d < mindist )
      {
        mindist = d;
        ibest = i;
      }	
    }
    return ibest;
  }
};

class CDEM
{
  const char                 *dempath;
  short int                  ****n, **o, *flatrow, **flatpatch;
  vector<vector <DATATYPE> >  elev;

  public:
  int        res;

  CDEM ( string dempath )
  {
    n = (short int****) calloc ( 180, sizeof(short int***) );
    *n = (short int***) calloc ( 180*360, sizeof(short int**) );
    for ( int i = 1; i < 180; ++i )  n[i] = n[i-1]+360;
    res = 0;
    o = flatpatch = NULL;
    this->dempath = dempath.c_str();
  }

  CDEM ( string filename, int cols )
  {
    res = 1;	  
    FILE  *fp; 
    DATATYPE *buf = (DATATYPE*) calloc ( cols, sizeof(DATATYPE) );
    if ( ( fp = fopen ( filename.c_str(), "rb" ) ) == NULL  )
    {
      printf ( "Error: File %s not found.\n", filename.c_str() );
      exit ( -1 );
    }
    fseek ( fp, 0, SEEK_END );
    int len = ftell ( fp );
    int rows = len/(cols*sizeof(DATATYPE));
    if ( rows*cols*sizeof(DATATYPE) != len )
    {
      printf ( "Error: File %s has inconsistent length.\n", filename.c_str() );
      exit ( -1 );
    }
    fseek ( fp, 0, SEEK_SET );
    elev.resize(rows);
    for ( int i = rows-1; i >= 0; --i )
    {
      elev[i].resize(cols);
      fread ( &elev[i][0], sizeof(DATATYPE), cols, fp );
    }
    printf ( "Read file %s with %i rows and %i columns\n", filename.c_str(), rows, cols );
    fclose ( fp );
  }

  void write ( string filename )
  {
    FILE  *fp;
    if ( ( fp = fopen ( filename.c_str(), "wb" ) ) != NULL  )
    {
      int byteswritten = 0;	    
      for ( int i = elev.size()-1; i >= 0; --i )
        byteswritten += fwrite ( &elev[i][0], sizeof(DATATYPE), elev[i].size(), fp );
      if ( byteswritten == elev.size()*elev[0].size() )
        printf ( "Successfully wrote file %s.\n", filename.c_str() );
      else
        printf ( "Warning: File %s could not be written.\n", filename.c_str() );
      fclose ( fp );
    }
    else
      printf ( "Warning: File %s could not be opened for writing.\n", filename.c_str() );
  }

  void readPatch ( double lon, double lat )
  {	  
    char  filename[1024];
    FILE  *fp; 

    lon = floor(lon);
    lat = floor(lat);
    sprintf ( filename, "%s/%c%s%g%c%s%s%g.hgm", dempath,
              lat >= 0 ? 'N' : 'S', fabs(lat) < 10 ? "0" : "", fabs(lat),
              lon >= 0 ? 'E' : 'W', fabs(lon) < 10 ? "0" : "", fabs(lon) < 100 ? "0" : "", fabs(lon) );
    if ( 1 || ( fp = fopen ( filename, "rb" ) ) == NULL  )
    {
      sprintf ( filename, "%s/%c%s%g%c%s%s%g.hgt", dempath,
                lat >= 0 ? 'N' : 'S', fabs(lat) < 10 ? "0" : "", fabs(lat),
                lon >= 0 ? 'E' : 'W', fabs(lon) < 10 ? "0" : "", fabs(lon) < 100 ? "0" : "", fabs(lon) );
      fp = fopen ( filename, "rb" );
    }
    if ( fp != NULL )
    {
      if ( !res )
      {
        fseek ( fp, 0, SEEK_END );
        int filesize = ftell ( fp );
        rewind ( fp );
        res = sqrt(filesize/2)-0.5;
        printf ( "Setting DEM resolution to %i points per degree\n",
                 res );
      }
      if ( flatpatch == NULL )
      {
        flatrow = (short int*) calloc ( res, sizeof(short int) );
        flatpatch = (short int**) calloc ( res, sizeof(short int*) );
        for ( int i = 0; i < res; ++i )  flatpatch[i] = flatrow;
      }
      int row = 90+lat;
      int col = 180+lon;
      n[row][col] = (short int**) calloc ( res, sizeof(short int*) );
      n[row][col][0] = (short int*) calloc ( res*res, sizeof(short int) );
      for ( int i = 1; i < res; ++i )  n[row][col][i] = n[row][col][i-1] + res;
      char *buf = (char*) calloc ( 2*(res+1)*(res+1), 1 );
      if ( fread ( buf, 1, 2*(res+1)*(res+1), fp ) == 2*(res+1)*(res+1) )
      {
        CNVal lsb, msb;
        char *p = buf+2*(res+1);  
        for ( int i = 0; i < res; ++i )
        {
          for ( int j = 0; j < res; ++j )
          {
            n[row][col][res-1-i][j] = (256*(signed char)p[1]+(unsigned char)p[0])&16383;
	    if ( n[row][col][res-1-i][j] > NODATA )  lsb += n[row][col][res-1-i][j]; 

	    n[row][col][res-1-i][j] = 256*(signed char)p[0]+(unsigned char)p[1];
	    if ( n[row][col][res-1-i][j] > NODATA )  msb += n[row][col][res-1-i][j]; 
	    p += 2;
          }
          p += 2;	  
        }
//printf ( "LSB %e MSB %e\n", lsb.getVar(), msb.getVar() );
	printf ( "Successfully read file %s in ", filename );
        if ( lsb.getVar() < msb.getVar() )
	{
	  printf ( "LSB byte order\n" );	
          p = buf+2*(res+1);  
          for ( int i = 0; i < res; ++i )
          {
            for ( int j = 0; j < res; ++j )
	    {
              n[row][col][res-1-i][j] = (256*(signed char)p[1]+(unsigned char)p[0])&16383;
	      p += 2;
	    }
            p += 2;
	  }	    
        }
        else  printf ( "MSB byte order\n" );	
      }
      else
        printf ( "Warning: Invalid file %s.\n", filename );
      free ( buf );
      fclose ( fp );
    }
    else  printf ( "File %s not found.\n", filename );
  }

  double getHeight ( int lon, int lat )
  {
// lon and lat in units of DEM pixels
    if ( scale < 1e5 )
      return  lat >= 0 && lat < elev.size() && lon >= 0 && lon < elev[lat].size() ?
	      elev[lat][lon] : -DBL_MAX; 
    int row = (lat+res*90)/res;
    int col = (lon+res*180)/res;
    if ( n[row][col] == NULL )  readPatch ( col-180, row-90 ); 
    return n[row][col][(lat+res*90)%res][(lon+res*180)%res];
  }

  double getSurfaceHeight ( int lon, double lat )
  {
// lon and lat in units of DEM pixels
    int ilat = floor(lat);
    double h1 = getHeight(lon,ilat);
    double h2 = getHeight(lon,ilat+1);
    return  h1 <= NODATA || h2 <= NODATA ? -DBL_MAX : (lat-ilat)*h2+(ilat+1-lat)*h1;
  }

  double getSurfaceHeight ( double lon, double lat )
  {
// lon and lat in units of DEM pixels
    int ilon = floor(lon);
    double h1 = getSurfaceHeight(ilon,lat);
    double h2 = getSurfaceHeight(ilon+1,lat); 
//printf ( "Outer interpolation: %f %i %f %f -> %f\n",
//    lon, ilon, h1, h2, ((lon-ilon)*h2+(ilon+f-lon)*h1)/f );
    return (lon-ilon)*h2+(ilon+1-lon)*h1;
  }

  void setHeight ( int lon, int lat, double h )
  {
// lon and lat in units of DEM pixels
    if ( scale < 1e5 && lat >= 0 && lat < elev.size() && lon >= 0 && lon < elev[lat].size() )
      elev[lat][lon] = h;
  }

  void setHeight ( double h )
  {
    if ( scale < 1e5 )
      for ( int lat = 0; lat < elev.size(); ++lat )
	for ( int lon = 0; lon < elev[lat].size(); ++lon )  elev[lat][lon] = h;
  }
};

template <class T>

class Range
{
  public:
  T vmax, vmin;
};

template <class T>

class CVector
{
  int        minind, maxind;
  vector<T>  v;

  public:
  CVector ( )
  {
  }
	
  CVector ( int minind, int maxind )
  {
    this->minind = minind;
    this->maxind = maxind;
    v.resize(maxind-minind+1);
  }

  T& operator [] ( const int ind )
  {
    return v[ind-minind];
  }	  
};

template <class T>

class CArray 
{
  int  minrow, maxrow, mincol, maxcol;
  vector<CVector<T> >  v;

  public:	
  CArray ()
  {
  }
	
  CArray ( int minrow, int maxrow, int mincol, int maxcol )
  {
    this->minrow = minrow;
    this->maxrow = maxrow;
    this->mincol = mincol;
    this->maxcol = maxcol;
    v.resize(maxrow-minrow+1);
    for ( int i = 0; i < v.size(); ++i )  v[i] = *new CVector<T>(mincol,maxcol);
  }

  CVector<T>& operator [] ( const int row )
  {
    return v[row-minrow];
  }	  

  T get ( int row, int col )
  {
    return v[row-minrow][col-mincol];
  }	  
};

class BinSet
{
  public:
  int           n;
  double        left, right;
  vector<CNVal> v;
 
  void init ( double left, double right, double width )
  {
    this->left = left;
    this->right = right;
    n = 2*((int)((right-left)/(2*width)+0.5)) + 1;
    v.resize(n);
  }

  double getX ( int i )
  {
    return  left+(right-left)*i/(n-1);
  }

  CNVal& operator [] ( const int i )
  {
    return v[i];
  }  

  double getMean ( int i )
  {
    return  v[i].getMean();
  }

  double getVar ( int i )
  {
    return  v[i].getVar();
  }

  void add ( double x, double u )
  {
    int  i = 0.5+(x-left)/(right-left)*(n-1);
    if ( i < 0 || i >= n )  return;
    v[i] += u;
  }
};

int getNearest ( int lon, int lat, CDEM *dem, Track *track, CArray<unsigned short int> &nearest )
{
  int imax = 0, imin = track->points.size()-1;	    
  for ( int nlon = lon-1; nlon <= lon+1; ++nlon )
    for ( int nlat = lat-1; nlat <= lat+1; ++nlat )
    {
      int tmp = nearest[nlon][nlat];
      if ( tmp < 65535 )
      {	      
        if ( tmp < imin )  imin = tmp;
        if ( tmp > imax )  imax = tmp;
      }	
    } 
    if ( imin > imax )
    {
      imin = 0;
      imax = track->points.size()-1;
    }	
    Point  p;
    if ( scale > 1e5 )
    {	    
      p.lon = (double)lon/dem->res;
      p.lat = (double)lat/dem->res;
    }
    else
    {	    
      p.lon = lon*scale;
      p.lat = lat*scale;
    }
    int ibest = track->findNearestPoint ( p, imin-1, imax+1 );
    if ( ibest != nearest[lon][lat] )
    {	    
//      printf ( "%i %i %i %i %i -> %i\n", lon, lat, imin, imax, nearest[lon][lat], ibest );
      nearest[lon][lat] = ibest; 
      return 1;
    }  
  return 0;  
}	    

int main ( int argc, char **argv )
{
  int           istart = 0, iend = 0, normalize = 0, cols = 0,
		minlon, maxlon, minlat, maxlat;
  double        r = 0., len = DBL_MAX, w = 1000.;
  string        dempath = "./", out = "profile.csv", extra = "";
  Track         track;
  BinSet        b;
  CNVal         trackelev;
  Point         startp;
  CDEM          *dem, *distdem = NULL;

  if ( argc == 1 )
  {
    printf ( "This program creates generalized swath profiles from a DEM. The profile\n"
	     "is taken perpendicularly to an arbitrary line, e.g., a river profile.\n"
             "The following types of DEMs are supported:\n"
             "(1) DEMs on a latitude-longitude grid in the style of SRTM or ASTER GDEM.\n"
             "    Resolution must be 1 or 3 arc seconds, and the DEM must be split into\n"
             "    tiles of 1x1 degrees named like SRTM or ASTER GDEM files. The data\n"
             "    must be stored in 2 byte integer format (MSB or LSB).\n"
             "(2) DEMS on a equally-spaced x-y lattice containing floating-point elevation\n"
	     "    data. The elevation data must be stored in a single file in row-wise\n"
	     "    order, starting from the upper left corner. Data type is single precision\n"
	     "    float (4 byte), but can be changed by recompiling the code.\n"
	     "DEM type 1 is the default.\n"
	     "Usage: swathprofile [ -n -c COLUMNS -m RES -d PATH/FILE -w HALF_LENGTH ...\n"
	     "                      -r RES -s LON LAT -l LENGTH -o OUTPUTFILE ] TRACKFILE\n"
	     "Parameters:\n" 
	     "TRACKFILE (required) defines the reference line, so that the generalized\n"
	     "    swath profile is taken perpendicularly to this line, e.g., a river.\n"
	     "    Currently, .gpx, .kml, and .csv (lon lat or x y ) are supported.\n"
	     "    Coordinates must be in degrees for DEM type 1 and in meters for DEM\n"
	     "    type 2 (only .csv ist allowed here). In this case, all coordinates\n"
	     "    must refer to the lower left corner of the DEM.\n"
             "-n (optional) defines the mode so that all elevations are considered in\n"
	     "    relation to the elevation of the respective point on the track. The mean\n"
	     "    elevation of these values is finally added to the elevations.\n"
             "-c COLUMNS (optional) switches from DEM type 1 to type 2. The parameter\n"
	     "    COLUMNS defines the number of columns of the DEM file.\n"
	     "-m RES (OPTIONAL) defines the x and y resolution of the DEM in case of a\n"
	     "    rectangular floating point DEM. Default is 1 meter.\n"
	     "-d PATH (optional for DEM type 1, required in case for DEM type 2) defines\n"
	     "    the path where the DEM files are located if not the current working\n"
	     "    directory (type 1) or the name of the DEM file (type 2).\n"
	     "-w HALF_LENGTH (optional) is half of the length of the profile. Default\n"
	     "    is 1000 m.\n"
	     "-r RESOLUTION (optional) defines the spatial resolution of the profile.\n"
	     "    Default is the N-S resolution of the DEM.\n"
	     "-s LON and LAT (optional) defines the point where the evaluation of the\n"
	     "    track starts. Default is the first point of the track.\n"
	     "-l LENGTH is the length of the track to be considered. Default is the\n"
	     "    total length of the track.\n"
	     "-o OUTPUTFILE is the filename used for the profile data. Default is\n"
	     "    profile.csv. The columns are:\n"
	     "    1: coordinate along the profile axis (negative = left of the track)\n"
	     "    2: mean elevation\n"
	     "    3: standard deviation of the elevation\n"
	     "    4: mean-standard deviation\n"
	     "    5: mean+standard deviation\n"
	     "    6: minimum elevation\n" 
	     "    7: maximum elevation\n"
	     "Note: All parameters and values referring to lengths are given in\n"
	     "      meters, longitudes and latitudes in decimal degrees.\n" ); 
    exit ( 0 );
  }
  startp.lat = -100;
  for ( int i = 1; i < argc; ++i )
    if ( argv[i][0] == '-' )
      switch ( argv[i][1] )
      {
        case 'd': dempath = argv[++i];
                  break;
        case 'c': cols = atoi(argv[++i]);
                  break;
        case 'm': scale = atof(argv[++i]);
                  break;
        case 'n': normalize = 1;
                  break;
        case 'w': w = atof(argv[++i]);
                  break;
        case 'r': r = atof(argv[++i]);
                  break;
	case 's': startp.lon = atof(argv[++i]);
	          startp.lat = atof(argv[++i]);	  
		  break;
        case 'l': len = atof(argv[++i]);
                  break;
        case 'o': out = argv[++i];
                  break;
        case 'x': extra = argv[++i];
                  break;
      }
    else  track.read(argv[i]);
  if ( cols == 0 )  scale = 111111.;  
  track.computeLength();
  if ( startp.lat >= -90 )
    istart = track.findNearestPoint ( startp, 0, 65535 );
  printf ( "Starting point: lat = %g, lon = %g, number = %i, length = %g\n",
	    track.points[istart].lat, track.points[istart].lon, istart,
	    track.len[istart] );
  for ( iend = istart; iend < track.points.size()-1; ++iend )
    if ( track.len[iend]-track.len[istart] >= len )  break;
  printf ( "End point: lat = %g, lon = %g, number = %i, length = %g\n",
	    track.points[iend].lat, track.points[iend].lon, iend,
	    track.len[iend] );
  track.computeBBox ( istart, iend );
  if ( cols )
  {	  
    dem = new CDEM ( dempath, cols );
    if ( !extra.empty() )
    {
      distdem = new CDEM ( dempath, cols );
      distdem->setHeight(FLT_MAX);
    } 
    minlon = (track.ll.lon-w)/scale - 1;
    maxlon = (track.ur.lon+w)/scale + 2;
    minlat = (track.ll.lat-w)/scale - 1;
    maxlat = (track.ur.lat+w)/scale + 2;
  }
  else
  {	  
    dem = new CDEM ( dempath );
    dem->readPatch ( track.ll.lon, track.ll.lat ); 
    minlon = dem->res * ( track.ll.lon - w/(scale*cos(track.ur.lat DEG)) ) - 1;
    maxlon = dem->res * ( track.ur.lon + w/(scale*cos(track.ur.lat DEG)) ) + 2;
    minlat = dem->res * ( track.ll.lat - w/scale ) - 1;
    maxlat = dem->res * ( track.ur.lat + w/scale ) + 2;
  } 
  if ( r <= 0. )  r = scale/dem->res; 
/*
  Track strack = track.shorten(0.25*r,r);
  if ( strack.points.size() != track.points.size() ) 
  {
    track = strack;	  
    printf ( "Shortening the track to %i points.\n", track.points.size() );
  }
  track.computeBBox();
*/  

//    printf ( "%g %g %g %g\n", track.ll.lon, track.ll.lat, track.ur.lon, track.ur.lat );
//  printf ( "%i %i %i %i\n", minlon, maxlon, minlat, maxlat );
  CArray<unsigned short int> nearest = *new CArray<unsigned short int>(minlon-1,maxlon+1,minlat-1,maxlat+1);
  for ( int lon = minlon-1; lon <= maxlon+1; ++lon ) 
    for ( int lat = minlat-1; lat <= maxlat+1; ++lat )
      nearest[lon][lat] = 65535;
  for ( int i = 0; i < track.points.size(); ++i )
  {
    int  lon, lat;
    if ( scale > 1e5 )
    {	    
      lon = track.points[i].lon*dem->res+0.5;	    
      lat = track.points[i].lat*dem->res+0.5;	    
    }
    else
    {	    
      lon = track.points[i].lon/scale+0.5;	    
      lat = track.points[i].lat/scale+0.5;	    
    }
    if ( lon >= minlon-1 && lon <= maxlon+1 && lat >= minlat-1 && lat <= maxlat+1 )
    {
//      printf ( "Nearest %i %g %g %i %i\n", i, track.points[i].lon, track.points[i].lat, lon, lat );
      nearest[lon][lat] = i;
    }
  }

  int nchanges, niter = 0;
  do
  {
    nchanges = 0;
    for ( int lon = minlon; lon <= maxlon; ++lon ) 
      for ( int lat = minlat; lat <= maxlat; ++lat )
         nchanges += getNearest ( lon, lat, dem, &track, nearest );
    for ( int lon = maxlon; lon >= minlon; --lon ) 
      for ( int lat = maxlat; lat >= minlat; --lat )
         nchanges += getNearest ( lon, lat, dem, &track, nearest );
    printf ( "Iteration %i: %i changes\n", ++niter, nchanges );
    if ( !nchanges )  break;
    nchanges = 0;
    for ( int lon = maxlon; lon >= minlon; --lon ) 
      for ( int lat = minlat; lat <= maxlat; ++lat )
         nchanges += getNearest ( lon, lat, dem, &track, nearest );
    for ( int lon = minlon; lon <= maxlon; ++lon ) 
      for ( int lat = maxlat; lat >= minlat; --lat )
         nchanges += getNearest ( lon, lat, dem, &track, nearest );
    printf ( "Iteration %i: %i changes\n", ++niter, nchanges );
  }
  while ( nchanges );
  b.init ( -w, w, r );
  for ( int lon = minlon; lon <= maxlon; ++lon ) 
    for ( int lat = minlat; lat <= maxlat; ++lat )
    {
      Point  p, c;
      if ( scale > 1e5 )
      {	    
        p.lon = (double)lon/dem->res;
        p.lat = (double)lat/dem->res;
      }
      else
      {	    
        p.lon = lon*scale;
        p.lat = lat*scale;
      }
//      printf ( "%i\n", nearest[lon][lat] );
//      printf ( "Nearest %i\n", track.findNearestPoint ( p, nearest[lon][lat]-1, nearest[lon][lat]+2 ) );
      if ( nearest[lon][lat] < istart || nearest[lon][lat] > iend )  continue;
      double dist = track.distance ( p, c, nearest[lon][lat]-2, nearest[lon][lat]+3 );
//      double dist2 = track.distance ( p, c, 0, 65535 );
//      if ( dist != dist2 )
//	printf ( "Inconsistent distance: %i %i %f %f\n", lon, lat, dist, dist2 );      
      if ( fabs(dist) >= w )  continue;
      if ( distdem != NULL )
//      {
        distdem->setHeight(lon,lat,dist);	      
//        printf ( "%i %i %e\n", lon, lat, dist );
//      }
      DATATYPE  h = dem->getHeight ( lon, lat );
      double  href = normalize ? dem->getSurfaceHeight ( dem->res*c.lon, dem->res*c.lat ) : 0.;    
//      if ( fabs(dist) <= r )  printf ( "%f %f %f %i %i %f %f\n", p.lon, p.lat, dist, h, nearest[lon][lat], c.lon, c.lat );
      if ( h > NODATA && href > NODATA )  b.add(dist,h-href);
      if ( fabs(dist) <= r )  trackelev += href;
    } 
  FILE *fp = fopen(out.c_str(),"w");
  for ( int i = 0; i < b.n; ++i )
  {
    double e = b[i].getMean() + trackelev.getMean(); 	  
    double std = sqrt(b[i].getVar());	  
    fprintf ( fp, "%g\t%g\t%g\t%g\t%g\t%g\t%g\n", b.getX(i), e, std, e-std, e+std, b[i].getMin()+trackelev.getMean(), b[i].getMax()+trackelev.getMean() );
  }
  fclose ( fp );
  if ( distdem != NULL )  distdem->write(extra);
  exit ( 0 );
}  



