
/*	

	SpaceTimeAutoRegistration:  Automated processing for Terrestrial Laser Scanning data 
	Copyright (C) 2017 Ryan Kromer (ryan.kromer@queensu.ca)
	This code is distributed under the terms of the GNU General Public License

	Point Cloud Library (PCL) - www.pointclouds.org
	Copyright (c) 2009-2012, Willow Garage, Inc.
	Copyright (c) 2012-, Open Perception, Inc
	Software License Agreement (BSD License)

	SpaceTimeAutoRegistration is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>

*/


#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#define _USE_MATH_DEFINES
#include <math.h>
#include <cmath>
#include <omp.h>
#include <stdio.h>
#include <pcl/common/angles.h>
#include <pcl/octree/octree_pointcloud_voxelcentroid.h>
#include <pcl/common/common.h>
#include <pcl/common/pca.h>
#include <pcl/io/pcd_io.h>
#include <pcl/point_types.h>
#include <pcl/features/normal_3d.h>
#include <pcl/console/time.h>
#include <pcl/console/print.h>
#include <pcl/console/parse.h>
#include <pcl/PointIndices.h>
#include "pcl/common/impl/centroid.hpp"
#include <pcl/impl/point_types.hpp>
#include <pcl/common/transforms.h>
#include <pcl/filters/passthrough.h>
#include <pcl/registration/icp.h>
#include <pcl/registration/correspondence_estimation.h>
#include <pcl/registration/correspondence_estimation_backprojection.h>
#include <pcl/registration/correspondence_estimation_normal_shooting.h>
#include <pcl/registration/ia_ransac.h>
#include <pcl/Keypoints/iss_3d.h>
#include <pcl/registration/correspondence_rejection_distance.h>
#include <pcl/registration/correspondence_rejection_median_distance.h>
#include <pcl/registration/correspondence_rejection_surface_normal.h>
#include <pcl/registration/correspondence_rejection.h>
#include <pcl/registration/correspondence_rejection_one_to_one.h>
#include <pcl/registration/correspondence_rejection_sample_consensus.h>
#include <pcl/registration/correspondence_rejection_trimmed.h>
#include <pcl/registration/transformation_estimation_lm.h>
#include <pcl/registration/transformation_estimation_point_to_plane.h>
#include <pcl/registration/transformation_estimation_point_to_plane_lls.h>
#include <pcl/registration/transformation_estimation_svd_scale.h>
#include <pcl/common/time.h>
#include <pcl/features/normal_3d_omp.h>
#include <pcl/registration/default_convergence_criteria.h>
#include <pcl/registration/icp.h>
#include <pcl/registration/sample_consensus_prerejective.h>
#include <pcl/segmentation/sac_segmentation.h>
#include <pcl/visualization/pcl_visualizer.h>
#include <pcl/filters/statistical_outlier_removal.h>
#include <pcl/octree/octree.h>
#include <pcl/octree/octree_pointcloud.h>
#include <pcl/filters/voxel_grid.h>
#include <pcl/features/fpfh_omp.h>
#include <pcl/filters/voxel_grid.h>
#include <QDir> 
#include <Eigen/Core>
#include <Eigen/dense> 
#include <pcl/visualization/registration_visualizer.h>
#include <pcl/surface/mls.h>
#include <pcl/surface/poisson.h>
#include <pcl/io/vtk_io.h>
using namespace std;
using namespace pcl;
using namespace pcl::io;
using namespace pcl::console;
typedef pcl::visualization::PointCloudColorHandlerCustom<pcl::PointXYZ> ColorHandlerT;


float avg (std::vector<float> v)
{
	float return_value ;
	int n = v.size();
	//return_value = v[ (n/2) + 1 ];// median
	return_value=std::accumulate(v.begin(),v.end(),0.0000)/v.size(); 
	return (return_value);
}

pcl::PointXYZ normalize( pcl::PointXYZ a , pcl::PointXYZ b , double mag)
{ 
	pcl::PointXYZ vector;
	vector.x = (b.x - a.x)/mag;
	vector.y = (b.y - a.y)/mag;
	vector.z = (b.z - a.z)/mag;

	return vector;
}
// define dot product 
float dot_normpt(pcl::Normal vector1 , pcl::PointXYZ vector2)
{
	float sum = 0;

	sum = (vector1.normal_x * vector2.x) + (vector1.normal_y * vector2.y) + (vector1.normal_z * vector2.z);

	return sum; 

}

void removeColumn(Eigen::MatrixXf& matrix, unsigned int colToRemove) // for removing column (memory management)
{
	unsigned int numRows = matrix.rows();
	unsigned int numCols = matrix.cols()-1;

	if( colToRemove < numCols )
		matrix.block(0,colToRemove,numRows,numCols-colToRemove) = matrix.block(0,colToRemove+1,numRows,numCols-colToRemove);
	matrix.conservativeResize(numRows,numCols+1);
}

pcl::PointCloud<pcl::PointXYZ>::Ptr boxClip ( pcl::PointCloud<pcl::PointXYZ>::Ptr cloud)
{
	// pass through filter, requires clopSettings .txt file 
	float xl,xu,yl,yu,zl,zu;

	string path = "clipSettings.txt"; //Specific to study case 
	ifstream fin;
	fin.open(path);  
	if(fin.is_open())                      // If it opened successfully
	{
		fin >> xl >> xu >> yl >> yu >> zl >> zu;  // Read the values and
		// store them in these variables
		fin.close();                   // Close the file
	}

	pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_x (new pcl::PointCloud<pcl::PointXYZ>);
	pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_y (new pcl::PointCloud<pcl::PointXYZ>);
	pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_filtered (new pcl::PointCloud<pcl::PointXYZ>);
	// Create the filtering object
	pcl::PassThrough<pcl::PointXYZ> pass;
	pass.setInputCloud (cloud);
	pass.setFilterFieldName ("x");
	pass.setFilterLimits (xl, xu);
	pass.filter (*cloud_x);

	//pcl::PassThrough<pcl::PointXYZ> pass;
	pass.setInputCloud (cloud_x);
	pass.setFilterFieldName ("y");
	pass.setFilterLimits (yl, yu);
	pass.filter (*cloud_y);

	//pcl::PassThrough<pcl::PointXYZ> pass;
	pass.setInputCloud (cloud_y);
	pass.setFilterFieldName ("z");
	pass.setFilterLimits (zl, zu);
	pass.filter (*cloud_filtered);

	return cloud_filtered ;
}

void
printHelp (int, char **argv)
{
  print_error ("Syntax is: %s input.xyz output.pcd\n", argv[0]);
}

bool
loadCloud (const string &filename, PointCloud<PointXYZ> &cloud)
{
  ifstream fs;
  fs.open (filename.c_str (), ios::binary);
  if (!fs.is_open () || fs.fail ())
  {
    PCL_ERROR ("Could not open file '%s'! Error : %s\n", filename.c_str (), strerror (errno)); 
    fs.close ();
    return (false);
  }
  
  string line;
  vector<string> st;

  while (!fs.eof ())
  {
    getline (fs, line);
    // Ignore empty lines
    if (line == "")
      continue;

    // Tokenize the line
    boost::trim (line);
    boost::split (st, line, boost::is_any_of ("\t\r, "), boost::token_compress_on);

    cloud.push_back (PointXYZ (float (atof (st[0].c_str ())), float (atof (st[1].c_str ())), float (atof (st[2].c_str ()))));
  }
  fs.close ();

  cloud.width = uint32_t (cloud.size ()); cloud.height = 1; cloud.is_dense = true;
  return (true);
}

bool
loadCloud (const string &filename, PointCloud<PointXYZI> &cloud)
{
	ifstream fs;
	fs.open (filename.c_str (), ios::binary);
	if (!fs.is_open () || fs.fail ())
	{
		PCL_ERROR ("Could not open file '%s'! Error : %s\n", filename.c_str (), strerror (errno)); 
		fs.close ();
		return (false);
	}

	string line;
	vector<string> st;

	while (!fs.eof ())
	{
		getline (fs, line);
		// Ignore empty lines
		if (line == "")
			continue;

		// Tokenize the line
		boost::trim (line);
		boost::split (st, line, boost::is_any_of ("\t\r, "), boost::token_compress_on);

		pcl::PointXYZI point; 
		point.x= float (atof (st[0].c_str ())); 
		point.y= float (atof (st[1].c_str ())); 
		point.z= float (atof (st[2].c_str ())); 
		point.intensity= float (atof (st[3].c_str ())); 
		cloud.points.push_back(point);
	}
	fs.close ();

	cloud.width = uint32_t (cloud.size ()); cloud.height = 1; cloud.is_dense = true;
	return (true);
}

double
computeCloudResolution(const pcl::PointCloud<pcl::PointXYZ>::ConstPtr& cloud)
{
	double resolution = 0.0;
	int numberOfPoints = 0;
	int nres;
	std::vector<int> indices(2);
	std::vector<float> squaredDistances(2);
	pcl::search::KdTree<pcl::PointXYZ> tree;
	tree.setInputCloud(cloud);

	for (size_t i = 0; i < cloud->size(); ++i)
	{
		if (! pcl_isfinite((*cloud)[i].x))
			continue;

		// Considering the second neighbor since the first is the point itself.
		nres = tree.nearestKSearch(i, 2, indices, squaredDistances);
		if (nres == 2)
		{
			resolution += sqrt(squaredDistances[1]);
			++numberOfPoints;
		}
	}
	if (numberOfPoints != 0)
		resolution /= numberOfPoints;

	return resolution;
}


void voxelFilter (pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud,const float &leafsize) 
{
	pcl::VoxelGrid<pcl::PointXYZ> sor3;
	sor3.setInputCloud(cloud);
	sor3.setLeafSize (leafsize, leafsize, leafsize);
	sor3.filter (*cloud);
	cout << cloud->size() << endl;
}

 pcl::PointCloud<pcl::Normal>::Ptr normalCalcFunction (const pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud_ref, double RAD)
 {
	 // Create the normal estimation class, and pass the input dataset to it

	 pcl::NormalEstimationOMP<pcl::PointXYZ, pcl::Normal> ne;
	 ne.setViewPoint(0,0,0);
	 ne.setInputCloud (cloud_ref);
	 cout << cloud_ref->size()<< endl;
	 // Create an empty kdtree representation, and pass it to the normal estimation object.
	 // Its content will be filled inside the object, based on the given input dataset 
	 pcl::search::KdTree<pcl::PointXYZ>::Ptr tree (new pcl::search::KdTree<pcl::PointXYZ> ());
	 ne.setSearchMethod (tree);
	 // Output datasets
	 pcl::PointCloud<pcl::Normal>::Ptr cloud_normals (new pcl::PointCloud<pcl::Normal>);
	 // Use all neighbors in a sphere of radius RAD
	 ne.setRadiusSearch (RAD);
	 ne.setNumberOfThreads(8);
	 // Compute the features
	 ne.compute (*cloud_normals);

	 return cloud_normals;
 }


 pcl::PointCloud<pcl::Normal>::Ptr normalCalcFunctiondown (const pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud_ref, double RAD)
 {

	 pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_refdown (new pcl::PointCloud<pcl::PointXYZ>);
	 pcl::copyPointCloud(*cloud_ref, *cloud_refdown);
	 voxelFilter (cloud_refdown,0.6); // to generalize make this 8-10x mean resolution 

	 // viewpoint, make view point variable and pass into function otherwise it is 0,0,0
	 pcl::search::KdTree<pcl::PointXYZ>::Ptr tree (new pcl::search::KdTree<pcl::PointXYZ> ());
	 NormalEstimationOMP<PointXYZ, Normal> ne;
	 ne.setNumberOfThreads (8);
	 ne.setInputCloud (cloud_refdown);

	 // Compute Normals on downsampled version of cloud 
	 pcl::PointCloud<pcl::Normal>::Ptr cloud_normals_visdown (new pcl::PointCloud<pcl::Normal>);
	 ne.setSearchMethod (tree);
	 ne.setKSearch(RAD);
	 ne.compute (*cloud_normals_visdown);

	 //container for normals 
	 pcl::PointCloud<pcl::Normal>::Ptr cloud_normals_vis (new pcl::PointCloud<pcl::Normal>);
	 cloud_normals_vis->width  = cloud_ref->size();
	 cloud_normals_vis->height = 1;
	 cloud_normals_vis->points.resize (cloud_normals_vis->width * cloud_normals_vis->height);

	 pcl::PointCloud<pcl::PointNormal>::Ptr cloud_with_Normalsdown (new pcl::PointCloud<pcl::PointNormal>);
	 pcl::copyPointCloud(*cloud_refdown, *cloud_with_Normalsdown);
	 pcl::copyPointCloud(*cloud_normals_visdown, *cloud_with_Normalsdown);

	 pcl::search::KdTree<pcl::PointNormal>::Ptr ktree (new pcl::search::KdTree<pcl::PointNormal> ());
	 ktree->setInputCloud(cloud_with_Normalsdown);

	 pcl::PointNormal searchpoint;
	 std::vector<int> pointIdxNKNSearch(1);
	 std::vector<float> pointNKNSquaredDistance(1);

	 for (long ii=0; ii < cloud_ref->size();ii++)
	 {
		 searchpoint.x=cloud_ref->points[ii].x;
		 searchpoint.y=cloud_ref->points[ii].y;
		 searchpoint.z=cloud_ref->points[ii].z;
		 ktree->nearestKSearch(searchpoint,1,pointIdxNKNSearch,pointNKNSquaredDistance);
		 cloud_normals_vis->points[ii].normal_x= cloud_normals_visdown->points[pointIdxNKNSearch[0]].normal_x;
		 cloud_normals_vis->points[ii].normal_y= cloud_normals_visdown->points[pointIdxNKNSearch[0]].normal_y;
		 cloud_normals_vis->points[ii].normal_z= cloud_normals_visdown->points[pointIdxNKNSearch[0]].normal_z;

	 }

	 return cloud_normals_vis;

 }

 void FilterInitial (pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud)  
 {

	 pcl::StatisticalOutlierRemoval<pcl::PointXYZ> sor;
	 sor.setInputCloud (cloud);
	 sor.setMeanK (100);
	 sor.setStddevMulThresh (1.0);
	 sor.filter (*cloud);

	 // Create the filtering object
	 pcl::PassThrough<pcl::PointXYZ> pass;
	 pass.setInputCloud (cloud);
	 pass.setFilterFieldName ("x");
	 pass.setFilterLimits (-1000,1000);
	 pass.filter (*cloud);

	 //pcl::PassThrough<pcl::PointXYZ> pass;
	 pass.setInputCloud (cloud);
	 pass.setFilterFieldName ("y");
	 pass.setFilterLimits (800, 1400);
	 pass.filter (*cloud);

	 //pcl::PassThrough<pcl::PointXYZ> pass;
	 pass.setInputCloud (cloud);
	 pass.setFilterFieldName ("z");
	 pass.setFilterLimits (-1000, 1000);
	 pass.filter (*cloud);

 }

 pcl::PointCloud<pcl::PointXYZ>::Ptr keypointsCalc (const PointCloud<PointXYZ>::Ptr &cloud){
	 pcl::PointCloud<pcl::PointXYZ>::Ptr keypoints (new pcl::PointCloud<pcl::PointXYZ>());
	 search::KdTree<PointXYZ>::Ptr search_method_ptr = search::KdTree<PointXYZ>::Ptr (new search::KdTree<PointXYZ>); 
	 pcl::ISSKeypoint3D<pcl::PointXYZ, pcl::PointXYZ> iss_detector;
	 iss_detector.setSalientRadius (0.5); 
	 iss_detector.setNonMaxRadius (1);
	 iss_detector.setInputCloud (cloud);
	 iss_detector.setSearchMethod(search_method_ptr);

	 double resolution = computeCloudResolution(cloud);
	 // Set the radius of the spherical neighborhood used to compute the scatter matrix.
	 iss_detector.setSalientRadius(5*resolution);
	 //cout << resolution << endl;
	 // Set the radius for the application of the non maxima supression algorithm.
	 iss_detector.setNonMaxRadius(3* resolution);
	 // Set the minimum number of neighbors that has to be found while applying the non maxima suppression algorithm.
	 iss_detector.setMinNeighbors(5);
	 // Set the upper bound on the ratio between the second and the first eigenvalue.
	 iss_detector.setThreshold21(0.975);
	 // Set the upper bound on the ratio between the third and the second eigenvalue.
	 iss_detector.setThreshold32(0.975);
	 // Set the number of prpcessing threads to use. 0 sets it to automatic.
	 iss_detector.setNumberOfThreads(8);
	 iss_detector.compute(*keypoints);

	 return keypoints;
 }

  void	roughAlign (const pcl::PointCloud<pcl::PointXYZI>::Ptr &cloud_ref,pcl::PointCloud<pcl::PointXYZI>::Ptr &cloud_data)
  {

	  pcl::console::TicToc tt;
	  tt.tic();
	  pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_data_filtered (new pcl::PointCloud<pcl::PointXYZ>);
	  pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_ref_filtered (new pcl::PointCloud<pcl::PointXYZ>);

	  double resolution=computeCloudResolution(cloud_data);
	  float leafsize=20*resolution;// tuned to study case
	  double RAD =120*resolution; //tuned to study case 

	  pcl::copyPointCloud(*cloud_data,*cloud_data_filtered);
	  pcl::copyPointCloud(*cloud_ref,*cloud_ref_filtered);

	  //Option to used voxel filter or kepypoints 
	  //voxelFilter(cloud_ref_filtered,leafsize);
	  //voxelFilter(cloud_data_filtered,leafsize);

	  pcl::PointCloud<pcl::Normal>::Ptr cloud_normals= normalCalcFunction (cloud_ref_filtered, 0.5*RAD); //tuned to study case 
	  pcl::PointCloud<pcl::Normal>::Ptr cloud_data_normals=normalCalcFunction (cloud_data_filtered,0.5* RAD);

	  pcl::PointCloud<pcl::PointXYZ>::Ptr keypointsAlign (new pcl::PointCloud<pcl::PointXYZ>()) ;
	  keypointsAlign= keypointsCalc(cloud_ref_filtered);
	  pcl::PointCloud<pcl::PointXYZ>::Ptr keypointsAligndata (new pcl::PointCloud<pcl::PointXYZ>()) ;
	  keypointsAligndata= keypointsCalc(cloud_data_filtered);
	  cout <<  "Keypoints Calculated" << endl;	
	  std::cout << "No of ISS ref keypoints: " << keypointsAlign->size() << std::endl;
	  std::cout << "No of ISS data keypoint: " << keypointsAligndata->size() << std::endl;
	  //std::cout << "No of points: " << cloud_data_filtered->size() << std::endl;

	  tt.toc();
	  tt.toc_print();

	  tt.tic(); 
	  //Calculated fast point feature histograms 
	  pcl::PointCloud<pcl::FPFHSignature33>::Ptr descriptors (new pcl::PointCloud<pcl::FPFHSignature33> ());
	  pcl::FPFHEstimationOMP<pcl::PointXYZ, pcl::Normal, pcl::FPFHSignature33> fpfh;
	  fpfh.setNumberOfThreads(8); 
	  //fpfh.setInputCloud (cloud_ref_filtered); //Use for voxel filter
	  fpfh.setInputCloud (keypointsAlign);		//Use for keypoints 
	  fpfh.setInputNormals (cloud_normals);
	  pcl::search::KdTree<PointXYZ>::Ptr tree (new pcl::search::KdTree<PointXYZ>);
	  fpfh.setSearchMethod (tree);
	  fpfh.setRadiusSearch (2.5*RAD); //specific to study case 
	  fpfh.setSearchSurface(cloud_ref_filtered);
	  fpfh.compute(*descriptors);
	  pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_aligned (new pcl::PointCloud<pcl::PointXYZ>);
	  pcl::PointCloud<pcl::FPFHSignature33>::Ptr descriptorsdata (new pcl::PointCloud<pcl::FPFHSignature33> ());
	  //fpfh.setInputCloud (cloud_data_filtered);//Use for voxel filter
	  fpfh.setInputCloud (keypointsAligndata);//Use for keypoints
	  fpfh.setInputNormals (cloud_data_normals);
	  fpfh.setSearchSurface(cloud_data_filtered);
	  fpfh.compute(*descriptorsdata);
	  cout <<  "Features Calculated" << endl;	
	  tt.toc();
	  tt.toc_print();
	  tt.tic();

	  // Deterimine Correspondences 

	  boost::shared_ptr<pcl::Correspondences> correspsondences (new pcl::Correspondences);
	  pcl::registration::CorrespondenceEstimation<pcl::FPFHSignature33,pcl::FPFHSignature33> est;
	  est.setInputSource (descriptorsdata);
	  est.setInputTarget (descriptors);
	  est.determineCorrespondences (*correspsondences);
	  cout << "Correspsondences found" << endl;	
	  cout << correspsondences->size()  << endl;	

	  tt.toc();
	  tt.toc_print();
	  tt.tic();

	  pcl::registration::CorrespondenceRejectorSampleConsensus<pcl::PointXYZ> rejector;
	  //rejector.setInputCloud (cloud_data_filtered); //Use with voxel filter
	  rejector.setInputCloud (keypointsAligndata); //Use with keypoints

	  //rejector.setInputTarget (cloud_ref_filtered);// Use with voxel filter 
	  rejector.setInputTarget (keypointsAlign); //Use with keypoints 

	  rejector.setInputCorrespondences (correspsondences);
	  rejector.setMaxIterations (200000);
	  rejector.setInlierThreshold (0.5);
	  //rejector.setRefineModel(true);
	  rejector.getRemainingCorrespondences(*correspsondences,*correspsondences);

	  if  (correspsondences->size()==0)

	  {
		  rejector.setInlierThreshold (2); // specific to study case
		  rejector.getCorrespondences (*correspsondences);

	  }

	  tt.toc();
	  tt.toc_print();

	  Eigen::Matrix4f transformation;
	  transformation=rejector.getBestTransformation();

	  pcl::transformPointCloud (*cloud_data, *cloud_data, transformation);

	  /*printf ("\n");
	  pcl::console::print_info ("    | %6.3f %6.3f %6.3f | \n", transformation (0,0), transformation (0,1), transformation (0,2));
	  pcl::console::print_info ("R = | %6.3f %6.3f %6.3f | \n", transformation (1,0), transformation (1,1), transformation (1,2));
	  pcl::console::print_info ("    | %6.3f %6.3f %6.3f | \n", transformation (2,0), transformation (2,1), transformation (2,2));
	  pcl::console::print_info ("\n");
	  pcl::console::print_info ("t = < %0.3f, %0.3f, %0.3f >\n", transformation (0,3), transformation (1,3), transformation (2,3));
	  pcl::console::print_info ("\n");*/

  }

void
	alignCloudsTran ( pcl::PointCloud<pcl::PointXYZ>::Ptr &cloudd,const  pcl::PointCloud<pcl::PointXYZ>::Ptr &cloudr){

		pcl::PointCloud<pcl::PointXYZ>::Ptr cloud1 (new pcl::PointCloud<pcl::PointXYZ>);
		pcl::PointCloud<pcl::PointXYZ>::Ptr cloud2 (new pcl::PointCloud<pcl::PointXYZ>);
		*cloud1=*cloudd;
		*cloud2=*cloudr;

		double resolution=computeCloudResolution(cloudr);

		voxelFilter (cloud1,10*resolution);
		voxelFilter (cloud2,10*resolution);

		resolution=computeCloudResolution(cloud1);

		pcl::console::TicToc tt;
		tt.tic(); 

		pcl::PointCloud<pcl::Normal>::Ptr cloud_normals= normalCalcFunctiondown (cloud2, 40);
		pcl::PointCloud<pcl::Normal>::Ptr cloud_data_normals=normalCalcFunctiondown (cloud1, 40);

		tt.toc();
		tt.toc_print();

		pcl::PointCloud<pcl::PointNormal>::Ptr cloud1_with_Normals (new pcl::PointCloud<pcl::PointNormal>);
		pcl::PointCloud<pcl::PointNormal>::Ptr cloud2_with_Normals (new pcl::PointCloud<pcl::PointNormal>);
		pcl::PointCloud<pcl::PointNormal>::Ptr output_with_Normals (new pcl::PointCloud<pcl::PointNormal>);

		pcl::PointCloud<pcl::PointXYZ>::Ptr output (new pcl::PointCloud<pcl::PointXYZ>);
		pcl::concatenateFields(*cloud1, *cloud_data_normals, *cloud1_with_Normals);
		pcl::concatenateFields(*cloud2, *cloud_normals, *cloud2_with_Normals);
		*output_with_Normals=*cloud1_with_Normals;
		*output=*cloud1; 

		boost::shared_ptr<pcl::Correspondences> correspsondences (new pcl::Correspondences);
		Eigen::Matrix4f transformFinal (Eigen::Matrix4f::Identity ());

		boost::shared_ptr<pcl::Correspondences> corresps(new pcl::Correspondences);
		boost::shared_ptr<pcl::Correspondences> corresps_filtered(new pcl::Correspondences);

		Eigen::Matrix4f transform_res_from_LM(Eigen::Matrix4f::Identity ());

		int iteration; //Eigen::Matrix4f transform;
		pcl::registration::DefaultConvergenceCriteria<float> conv_crit (iteration, transform_res_from_LM,*corresps_filtered);
		//conv_crit.setMaximumIterationsSimilarTransforms (5);
		conv_crit.setMaximumIterations(20);
		//conv_crit.setRotationThreshold (cos (0.5 * M_PI / 180.0));
		//conv_crit.setAbsoluteMSE(0.000001);
		iteration=0;

		do 
		{

			iteration=iteration+1;
			pcl::registration::CorrespondenceEstimationNormalShooting<pcl::PointNormal, pcl::PointNormal, pcl::PointNormal> est;
			est.setInputSource(output_with_Normals); 
			est.setInputTarget(cloud2_with_Normals);
			est.setSourceNormals(output_with_Normals);
			//est.setTargetNormals(cloud2_with_Normals);
			est.setKSearch(10); 
			est.determineCorrespondences(*corresps);

			pcl::registration::CorrespondenceRejectorSampleConsensus<pcl::PointXYZ> rejector3;
			rejector3.setInputCloud (output);
			rejector3.setInputTarget (cloud2);
			//rejector3.setRefineModel(true);
			rejector3.setInputCorrespondences (corresps);
			rejector3.setMaxIterations (4000);
			rejector3.setInlierThreshold (0.3);
			rejector3.getCorrespondences (*corresps_filtered);
			pcl::registration::CorrespondenceRejectorSurfaceNormal::Ptr rej_normals (new pcl::registration::CorrespondenceRejectorSurfaceNormal);
			rej_normals->setInputCorrespondences(corresps_filtered);
			double degree=40.0;
			double threshold=acos (deg2rad (degree));
			rej_normals->setThreshold (threshold);
			rej_normals->initializeDataContainer<pcl::PointNormal, pcl::PointNormal> ();
			rej_normals->setInputSource<pcl::PointNormal> (output_with_Normals);
			rej_normals->setInputNormals<pcl::PointNormal, pcl::PointNormal> (output_with_Normals);
			rej_normals->setInputTarget<pcl::PointNormal> (cloud2_with_Normals);
			rej_normals->setTargetNormals<pcl::PointNormal, pcl::PointNormal> (cloud2_with_Normals);
			rej_normals->getCorrespondences (*corresps_filtered);

			pcl::registration::CorrespondenceRejectorMedianDistance::Ptr rejector2 (new pcl::registration::CorrespondenceRejectorMedianDistance);
			rejector2->setInputCorrespondences (corresps_filtered);
			rejector2->setMedianFactor(1.3);
			rejector2->getCorrespondences (*corresps_filtered);

			pcl::registration::TransformationEstimationPointToPlaneLLS<pcl::PointNormal, pcl::PointNormal,float> trans_est_lm;

			trans_est_lm.estimateRigidTransformation (*output_with_Normals, *cloud2_with_Normals,*corresps_filtered, transform_res_from_LM);

			transformFinal=transformFinal*transform_res_from_LM;

			pcl::transformPointCloud (*cloud1, *output, transformFinal);
			pcl::transformPointCloudWithNormals(*cloud1_with_Normals, *output_with_Normals, transformFinal);

		}
		while (!conv_crit.hasConverged ());

		pcl::transformPointCloud(*cloudd,*cloudd,transformFinal);

		*cloud1=*cloudd;
		*cloud2=*cloudr;

		voxelFilter (cloud1,2.5*resolution);
		voxelFilter (cloud2,2.5*resolution);

}
void
	alignCloudsFine (pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud1,const pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud2){

		pcl::PointCloud<pcl::Normal>::Ptr normals2= normalCalcFunction (cloud2, 3); 
		pcl::PointCloud<pcl::Normal>::Ptr normals1=normalCalcFunction (cloud1, 3);

		pcl::PointCloud<pcl::PointNormal>::Ptr cloud1_with_Normals (new pcl::PointCloud<pcl::PointNormal>);
		pcl::PointCloud<pcl::PointNormal>::Ptr cloud2_with_Normals (new pcl::PointCloud<pcl::PointNormal>);

		pcl::PointCloud<pcl::PointNormal>::Ptr output (new pcl::PointCloud<pcl::PointNormal>);

		pcl::concatenateFields(*cloud1, *normals1, *cloud1_with_Normals);
		pcl::concatenateFields(*cloud2, *normals2, *cloud2_with_Normals);

		boost::shared_ptr<pcl::Correspondences> correspsondences (new pcl::Correspondences);
		Eigen::Matrix4f transformFinal (Eigen::Matrix4f::Identity ());

		pcl::registration::CorrespondenceEstimation<pcl::PointXYZ, pcl::PointXYZ> est;

		est.setInputSource (cloud1);
		est.setInputTarget (cloud2);
		est.determineCorrespondences (*correspsondences);

		pcl::registration::TransformationEstimationLM<pcl::PointXYZ, pcl::PointXYZ> trans_est;
		Eigen::Matrix4f transform_res_rough;
		trans_est.estimateRigidTransformation (*cloud1,*cloud2, *correspsondences, transform_res_rough);
		transformFinal=transformFinal*transform_res_rough;

		pcl::transformPointCloud (*cloud1, *cloud1, transform_res_rough);
		pcl::transformPointCloudWithNormals(*cloud1_with_Normals, *cloud1_with_Normals, transform_res_rough);

		boost::shared_ptr<pcl::Correspondences> corresps(new pcl::Correspondences);
		boost::shared_ptr<pcl::Correspondences> corresps_filtered(new pcl::Correspondences);

		Eigen::Matrix4f transform_res_from_LM;

		int iteration; //Eigen::Matrix4f transform;
		pcl::registration::DefaultConvergenceCriteria<float> conv_crit (iteration, transform_res_from_LM,*corresps_filtered);
		//conv_crit.setMaximumIterationsSimilarTransforms (10);
		//conv_crit.setMaximumIterations(50);
		//conv_crit.setRotationThreshold (cos (0.5 * M_PI / 180.0));
		//conv_crit.setAbsoluteMSE(0.000001);
		iteration=0;

		do
		{
			iteration=iteration+1;
			pcl::registration::CorrespondenceEstimationNormalShooting<pcl::PointNormal, pcl::PointNormal, pcl::PointNormal> est;

			est.setInputSource(cloud1_with_Normals); 
			est.setInputTarget(cloud2_with_Normals);
			est.setSourceNormals(cloud1_with_Normals);
			est.determineCorrespondences(*corresps);

			pcl::registration::CorrespondenceRejectorSurfaceNormal::Ptr rej_normals (new pcl::registration::CorrespondenceRejectorSurfaceNormal);
			rej_normals->setInputCorrespondences(corresps);
			double degree=32.0;
			double threshold=acos (deg2rad (degree));
			rej_normals->setThreshold (threshold);
			rej_normals->initializeDataContainer<pcl::PointNormal, pcl::PointNormal> ();
			rej_normals->setInputSource<pcl::PointNormal> (cloud1_with_Normals);
			rej_normals->setInputNormals<pcl::PointNormal, pcl::PointNormal> (cloud1_with_Normals);
			rej_normals->setInputTarget<pcl::PointNormal> (cloud2_with_Normals);
			rej_normals->setTargetNormals<pcl::PointNormal, pcl::PointNormal> (cloud2_with_Normals);
			rej_normals->getCorrespondences (*corresps_filtered);

			pcl::registration::CorrespondenceRejectorMedianDistance::Ptr rejector2 (new pcl::registration::CorrespondenceRejectorMedianDistance);
			rejector2->setInputCorrespondences (corresps_filtered);
			rejector2->setMedianFactor(1);
			rejector2->getCorrespondences (*corresps_filtered);

			pcl::registration::TransformationEstimationPointToPlaneLLS<pcl::PointNormal, pcl::PointNormal,float> trans_est_lm;
			trans_est_lm.estimateRigidTransformation (*cloud1_with_Normals, *cloud2_with_Normals,*corresps_filtered, transform_res_from_LM);
			transformFinal=transformFinal*transform_res_from_LM;

			pcl::transformPointCloud (*cloud1, *cloud1, transform_res_from_LM);
			pcl::transformPointCloudWithNormals(*cloud1_with_Normals, *cloud1_with_Normals, transform_res_from_LM);


		}
		while (!conv_crit.hasConverged ());

}

 pcl::PointCloud<pcl::PointXYZI>::Ptr rawDistCalc (const pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud_data,const pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud_ref,pcl::PointCloud<pcl::Normal>::Ptr &cloud_normals, int K, double maxdist,double factor)
 {
	 //create data cloud tree 
	 pcl::search::KdTree<pcl::PointXYZ> kdtree_data;
	 kdtree_data.setInputCloud (cloud_data);
	 double resolutiondata = computeCloudResolution(cloud_data);

	 pcl::PointXYZ searchpoint;
	 std::vector<int> pointIdxNKNSearch(K);
	 std::vector<float> pointNKNSquaredDistance(K);
	 pcl::PointCloud<pcl::PointXYZ>::Ptr ShortDistVect (new pcl::PointCloud<pcl::PointXYZ>);
	 ShortDistVect->width = K;
	 ShortDistVect->height = 1;
	 ShortDistVect->points.resize (ShortDistVect->width * ShortDistVect->height);
	 std::vector<double> RawdistMag(cloud_ref->points.size ());

	 pcl::PointCloud<pcl::PointXYZI>::Ptr Rawdist (new pcl::PointCloud<pcl::PointXYZI>);
	 Rawdist->width = cloud_ref->points.size ();
	 Rawdist->height = 1;
	 Rawdist->points.resize (Rawdist->width * Rawdist->height);

	 //#pragma omp parallel for

	 for (long i = 0; i < cloud_ref->points.size (); ++i)
	 {
		 // search point 
		 searchpoint = cloud_ref->points[i]; 
		 kdtree_data.nearestKSearch (searchpoint, K, pointIdxNKNSearch, pointNKNSquaredDistance);
		 std::vector<float> dot_prod(K);
		 std::vector<float> cylinderRad;

		 for (long j = 0; j < pointIdxNKNSearch.size (); ++j)
		 {
			 ShortDistVect->points[j].x= cloud_data->points[pointIdxNKNSearch[j]].x - cloud_ref->points[i].x; 
			 ShortDistVect->points[j].y=  cloud_data->points[pointIdxNKNSearch[j]].y-cloud_ref->points[i].y; 
			 ShortDistVect->points[j].z=  cloud_data->points[pointIdxNKNSearch[j]].z-cloud_ref->points[i].z;
			 dot_prod[j] = dot_normpt(cloud_normals->points[i],ShortDistVect->points[j]); 

			 // temp is the rejection 
			 float temp= sqrt(pow((ShortDistVect->points[j].x-cloud_normals->points[i].normal_x*dot_prod[j]),2)+pow((ShortDistVect->points[j].y-cloud_normals->points[i].normal_y*dot_prod[j]),2)+pow((ShortDistVect->points[j].z-cloud_normals->points[i].normal_z*dot_prod[j]),2));

			 if (temp< factor*resolutiondata)
			 {
				 cylinderRad.push_back(dot_prod[j]);
			 }

		 }  

		 float average;

		 if (cylinderRad.empty()){
			 average= std::numeric_limits<double>::quiet_NaN();
		 }
		 else
		 {
			 average=avg(cylinderRad); 

			 if (average<-maxdist) 
			 {
				 average=std::numeric_limits<double>::quiet_NaN();
			 }

			 if (average>maxdist) 
			 {
				 average=std::numeric_limits<double>::quiet_NaN();
			 }
		 }
		 Rawdist->points[i].x=cloud_ref->points[i].x;
		 Rawdist->points[i].y=cloud_ref->points[i].y;
		 Rawdist->points[i].z=cloud_ref->points[i].z;
		 Rawdist->points[i].intensity=average;
	 }
	 cout << "Rawdone" << endl; 
	 return Rawdist;

 }

 pcl::PointCloud<pcl::PointXYZI>::Ptr filtDistCalc (int NN, pcl::PointCloud<pcl::PointXYZI>::Ptr &Rawdist,pcl::KdTreeFLANN<pcl::PointXYZ> &kdtree_ref,pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud_ref) 
 {		
	 std::vector<int> pointIdxNKNSearchNN(NN);
	 std::vector<float> pointNKNSquaredDistanceNN(NN);

	 pcl::PointXYZ searchpoint;

	 pcl::PointCloud<pcl::PointXYZI>::Ptr Filtdist (new pcl::PointCloud<pcl::PointXYZI>);
	 Filtdist->width = Rawdist->points.size ();
	 Filtdist->height = 1;
	 Filtdist->points.resize (Filtdist->width * Filtdist->height);

	 //#pragma omp parallel for
	 for (long ii = 0; ii < Rawdist->points.size (); ++ii)
	 {
		 searchpoint=cloud_ref->points[ii]; 
		 kdtree_ref.nearestKSearch (searchpoint, NN, pointIdxNKNSearchNN, pointNKNSquaredDistanceNN);
		 std::vector<float> Filt;
		 //cout << "line 484" << endl;

		 for (long jj = 0; jj < pointIdxNKNSearchNN.size (); ++jj)
		 {
			 if ((Rawdist->points[pointIdxNKNSearchNN[jj]].intensity)!=(Rawdist->points[pointIdxNKNSearchNN[jj]].intensity)) //if it is nan
			 {
				 //do nothing
			 }
			 else
			 {
				 Filt.push_back( Rawdist->points[pointIdxNKNSearchNN[jj]].intensity);
			 }

		 } 
		 float averageNN;

		 if (Filt.empty()){

			 averageNN= std::numeric_limits<double>::quiet_NaN();
		 }
		 else
		 {
			 averageNN=avg(Filt);
		 }

		 Filtdist->points[ii].x=Rawdist->points[ii].x;
		 Filtdist->points[ii].y=Rawdist->points[ii].y;
		 Filtdist->points[ii].z=Rawdist->points[ii].z;
		 Filtdist->points[ii].intensity=averageNN;
	 }
	 cout << "Filtdone" << endl; 
	 return Filtdist;

 }

  pcl::PointCloud<pcl::PointXYZI>::Ptr filtDistRad (double factorSearch,double resolutionref,pcl::PointCloud<pcl::PointXYZI>::Ptr &Rawdist,pcl::KdTreeFLANN<pcl::PointXYZ> &kdtree_ref,pcl::PointCloud<pcl::PointXYZ>::Ptr &cloud_ref) 
  {		

	  std::vector<int> pointIdxRad;
	  std::vector<float> pointSquaredDistRad;
	  pcl::PointXYZ searchpoint;
	  pcl::PointCloud<pcl::PointXYZI>::Ptr Filtdist (new pcl::PointCloud<pcl::PointXYZI>);
	  Filtdist->width = Rawdist->points.size ();
	  Filtdist->height = 1;
	  Filtdist->points.resize (Filtdist->width*Filtdist->height);

	  double radSearch=factorSearch*resolutionref; 
	  //#pragma omp parallel for
	  float areaCircle=M_PI*factorSearch*factorSearch; // according to Gauss circle problem the minimum amount of vertices of a grid inside a circle given a radius is pi-r^2 with E(r)=0. I am making sure I have enough points for NN averaging. 

	  for (long ii = 0; ii < Rawdist->points.size (); ++ii)
	  {
		  float averageNN;

		  searchpoint=cloud_ref->points[ii]; 

		  //  kdtree_ref.nearestKSearch (searchpoint, NN, pointIdxNKNSearchNN, pointNKNSquaredDistanceNN);
		  kdtree_ref.radiusSearch(searchpoint,radSearch,pointIdxRad, pointSquaredDistRad);
		  std::vector<float> Filt;
		  //cout << pointIdxRad.size() << endl;

		  if (pointIdxRad.size()< 0.167*areaCircle) //1/6 of expected points in ideal conditions grid overlying circle 
		  { 
			  averageNN= std::numeric_limits<double>::quiet_NaN();
		  } 

		  else
		  {
			  for (long jj = 0; jj <pointIdxRad.size (); ++jj)
			  {
				  if ((Rawdist->points[pointIdxRad[jj]].intensity)!=(Rawdist->points[pointIdxRad[jj]].intensity)) //if it is nan
				  {
					  //do nothing
				  }
				  else
				  {
					  Filt.push_back( Rawdist->points[pointIdxRad[jj]].intensity);
				  } 
			  } 

			  averageNN=avg(Filt);
		  } 


		  Filtdist->points[ii].x=Rawdist->points[ii].x;
		  Filtdist->points[ii].y=Rawdist->points[ii].y;
		  Filtdist->points[ii].z=Rawdist->points[ii].z;
		  Filtdist->points[ii].intensity=averageNN;
	  }
	  cout << "Filtdone" << endl; 
	  return Filtdist;

		}

void scannerTargetDistance(const pcl::PointCloud<pcl::PointXYZI>::Ptr &cloud_ref,pcl::PointCloud<pcl::PointXYZI>::Ptr &cloud_data, int kk) // Target correction specific to study case 

{
	pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_ref_target1 (new pcl::PointCloud<pcl::PointXYZI>);
	pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_ref_target2 (new pcl::PointCloud<pcl::PointXYZI>);
	pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_data_target1 (new pcl::PointCloud<pcl::PointXYZI>);
	pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_data_target2 (new pcl::PointCloud<pcl::PointXYZI>);

	pcl::PassThrough<pcl::PointXYZI> pass;

	// Correction for target 2 (post processed) 
	pcl::PointCloud<pcl::PointXYZ>::Ptr CorTarg(new pcl::PointCloud<pcl::PointXYZ>);
	loadCloud("Correction_Target.txt", *CorTarg) ;

	// Extract Target 1 
	float yl,yu;

	yl=985;
	yu=995;


	pass.setInputCloud (cloud_ref);
	pass.setFilterFieldName ("y");
	pass.setFilterLimits (yl, yu);
	pass.filter (*cloud_ref_target1);

	pass.setInputCloud (cloud_ref_target1);
	pass.setFilterFieldName ("intensity");
	pass.setFilterLimits (40, 10000);
	pass.filter (*cloud_ref_target1);

	pass.setInputCloud (cloud_data);
	pass.setFilterFieldName ("y");
	pass.setFilterLimits (yl, yu);
	pass.filter (*cloud_data_target1);

	pass.setInputCloud (cloud_data_target1);
	pass.setFilterFieldName ("intensity");
	pass.setFilterLimits (40, 10000);
	pass.filter (*cloud_data_target1);

	// Extract Target 2 - cible 9

	yl=1040;
	yu=1060;

	pass.setInputCloud (cloud_ref);
	pass.setFilterFieldName ("y");
	pass.setFilterLimits (yl, yu);
	pass.filter (*cloud_ref_target2);

	pass.setInputCloud (cloud_ref_target2);
	pass.setFilterFieldName ("intensity");
	pass.setFilterLimits (40, 10000);
	pass.filter (*cloud_ref_target2);

	pass.setInputCloud (cloud_data);
	pass.setFilterFieldName ("y");
	pass.setFilterLimits (yl, yu);
	pass.filter (*cloud_data_target2);

	pass.setInputCloud (cloud_data_target2);
	pass.setFilterFieldName ("intensity");
	pass.setFilterLimits (40, 10000);
	pass.filter (*cloud_data_target2);

	Eigen::Vector4f centroid_ref_target1;
	Eigen::Vector4f centroid_ref_target2;
	Eigen::Vector4f centroid_data_target1;
	Eigen::Vector4f centroid_data_target2;

	pcl::compute3DCentroid(*cloud_data_target1,centroid_data_target1);
	pcl::compute3DCentroid(*cloud_data_target2,centroid_data_target2);

	pcl::compute3DCentroid(*cloud_ref_target1,centroid_ref_target1);
	pcl::compute3DCentroid(*cloud_ref_target2,centroid_ref_target2);

	// corrected moving target
	centroid_ref_target2(1)=centroid_data_target1(1)+CorTarg->points[kk].x*0.866;
	centroid_ref_target2(2)=centroid_data_target1(2)+CorTarg->points[kk].x*0.5; 
	// distance between targets 

	float targetmeanN= sqrt(pow(centroid_ref_target1(0)-centroid_ref_target2(0),2)+pow(centroid_ref_target1(1)-centroid_ref_target2(1),2)+pow(centroid_ref_target1(2)-centroid_ref_target2(2),2))/sqrt(pow(centroid_data_target1(0)-centroid_data_target2(0),2)+pow(centroid_data_target1(1)-centroid_data_target2(1),2)+pow(centroid_data_target1(2)-centroid_data_target2(2),2));

	for (long i = 0; i < cloud_data->points.size (); ++i)
	{		
		cloud_data->points[i].x=cloud_data->points[i].x*targetmeanN; 
		cloud_data->points[i].y=cloud_data->points[i].y*targetmeanN; 
		cloud_data->points[i].z=cloud_data->points[i].z*targetmeanN; 
	}

}

 float Saturation_Vapor_Pressure(float t)
 {
	 //Assign Values to Constants

	 float k1=1.16705214528*pow(10.0f,3.0f);
	 float k2=-7.24213167032*pow(10.0f,5.0f);
	 float k3=-1.70738469401*10.0;
	 float k4=1.20208247025*pow(10.0f,4.0f);
	 float k5=-3.23255503223*pow(10.0f,6.0f);
	 float k6=1.49151086135*10.0;
	 float k7=-4.82326573616*pow(10.0f,3.0f);
	 float k8=4.05113405421*pow(10.0f,5.0f);
	 float k9=-2.38555575678*pow(10.0f,-1.0f);
	 float k10=6.50175348448*pow(10.0f,2.0f);

	 //Intermediate Calculations
	 float tk=t+273.15;
	 float omega=tk+(k9/(tk-k10));

	 float a=pow(omega,2.0f)+(k1*omega)+k2;

	 float b=k3*pow(omega,2.0f)+(k4*omega)+k5;
	 float c=k6*pow(omega,2.0f)+(k7*omega)+k8;
	 float x=-b+sqrt(pow(b,2.0f)-4.0*a*c);

	 // Final Result (the saturation vapour pressure)
	 float psv=pow(10.0f,6.0f)*pow((2.0f*c)/x,4.0f);
	 return psv; 
 }

 float Saturation_Vapor_Pressure_ice(float t)

 { 
	 //constants 
	 float a1=-13.928169;
	 float a2=34.7078238;
	 float tk=t+273.15; // temp Kelvin
	 float thi=tk/273.16;
	 float y=a1*(1-pow(thi,-1.5f))+a2*(1-pow(thi,-1.25f));
	 float psv =611.657*pow(exp(1.0f),y);

	 return psv ;
 } 

 float ciddor (float w,float c,float t,float p,float h)
 {
	 // w is vacuum wavelength of laser (micron) valid from 0.3 to 1.7 
	 // c is the co2 (ppm mole) valid from 0 to 2000, often 450 ppm (outside)
	 // t is air temperature (C), from -40 to 100 C 
	 // p is air pressure (Pa) from 10 000 to 140 000 typicallay 101 325 Pa 
	 //h is relative humidity (%) valid from 0 to 100 % 

	 //Assign Values to Constants
	 float w0=295.235;
	 float w1=2.6422;
	 float w2=-0.03238;
	 float w3=0.004028;
	 float k0=238.0185;
	 float k1=5792105;
	 float k2=57.362;
	 float k3=167917;

	 float a0=1.58123*(pow(10.0,-6.0));
	 float a1=-2.9331*(pow(10.0,-8.0));
	 float a2=1.1043*(pow(10.0,-10.0));
	 float b0=5.707*(pow(10.0,-6.0));
	 float b1=-2.051*(pow(10.0,-8.0));
	 float c0=1.9898*(pow(10.0,-4.0));
	 float c1=-2.376*(pow(10.0,-6.0));
	 float d=1.83*(pow(10.0,-11.0));
	 float e=-0.765*(pow(10.0,-8.0));


	 float pr1=101325;
	 float tr1=288.15;
	 float za=0.9995922115;
	 float r=8.314472;
	 float mv=0.018015;
	 //intermediate calculations
	 float s=1/(pow(w,2));
	 float ras= pow(10.0,-8.0)*((k1/(k0-s))+(k3/(k2-s)));
	 float rvs=1.022*pow(10.0,-8.0)*(w0+(w1*s)+(w2*pow(s,2.0f))+(w3*pow(s,3.0f)));

	 float ma=0.0289635*(1.2011*pow(10.0,-8.0)*(c-400));

	 float raxs=ras*(1+(5.34*pow(10.0,-7.0)*(c-450)));
	 float tk=t+273.15;

	 //calculation fo relative humidity 

	 float alpha=1.00062;
	 float beta=3.14*pow(10.0,-8.0); 
	 float gama=5.6*pow(10.0,-7.0); 

	 float f=alpha+(beta*p)+(gama*(pow(t,2.0f)));

	 //Determine whether over water or ice and call relevant fuction to calculate the saturation vapor preassure
	 float psv;

	 if (t>0)
	 {
		 psv=Saturation_Vapor_Pressure(t);
	 }
	 else

	 {
		 psv=Saturation_Vapor_Pressure_ice(t);
	 }

	 //Final Result (the humidity as mole fraction)

	 float x=((h/100.0)*f*psv)/p;

	 float zm1=(p/tk);

	 float zm2=a0+(a1*t)+a2*pow(t,2.0f);
	 float zm3=(b0+b1*t)*x;
	 float zm4=(c0+(c1*t))*pow(x, 2.0f);
	 float zm5=pow(p/ tk,2.0f)*(d+e*pow(x,2.0f));
	 float zm=1-(zm1*(zm2+zm3+zm4))+zm5;

	 //Calculate Zw
	 float pref2=1333;
	 float tref2=293.15;
	 float zw1=(pref2/tref2);
	 float zw2=a0+a1*(tref2-273.15)+a2*pow(tref2-273.15,2.0);

	 float zw3=(b0+(b1*(tref2-273.15)))*1;
	 float zw4=(c0+(c1*(tref2-273.15)))*pow(1.0,2.0);
	 float zw5=pow(pref2/tref2,2.0f)*(d+e*pow(1.0f,2.0f));

	 float zw=1-(zw1*(zw2+zw3+zw4))+zw5;

	 float rhoaxs=(pr1*ma)/(za*r*tr1);
	 float rhov=(x*p*ma)/(zm*r*tk)*(1-1*(1-mv/ma));
	 float rhoa=((1-x)*p*ma)/(zm*r*tk);
	 float rhovs=(pref2*ma/(zw*r*tref2))*(1-1*(1-mv/ma));

	 //Final Result (the refractive index)
	 float n=1+((rhoa/rhoaxs)*raxs)+((rhov/rhovs)*rvs);

	 return n; 

 }

 void tempCorrectModel(pcl::PointCloud<pcl::PointXYZI>::Ptr &cloud, pcl::PointCloud<pcl::PointXYZ>::Ptr &temp, int kk)
 {

		float w =1.064; //valid for optech ilris laser LR scanner 
		float c= 450; // outside CO2 content 450 ppm

		float t=temp->points[kk].x;
		float p=temp->points[kk].y;
		float h=temp->points[kk].z;
		cout<<" T P H:" << endl;
	    cout << t << endl;
		float n= ciddor(w,c,t,p,h);
		cout << "ATM corection:" << endl;
		cout << n << endl;

		for (long i = 0; i < cloud->points.size (); ++i)
			{		
				cloud->points[i].x=cloud->points[i].x*n; 
				cloud->points[i].y=cloud->points[i].y*n; 
				cloud->points[i].z=cloud->points[i].z*n; 
			}
 }

int
 main (int argc, char** argv)
{


	pcl::console::TicToc tt;
	tt.tic(); 


	pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_ref (new pcl::PointCloud<pcl::PointXYZI>);
	pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_data_align (new pcl::PointCloud<pcl::PointXYZ>);
	pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_ref_align (new pcl::PointCloud<pcl::PointXYZ>);
	pcl::PointCloud<pcl::PointXYZ>::Ptr cloud_ref_noI (new pcl::PointCloud<pcl::PointXYZ>);


	// Load weather data as a point cloud 
	pcl::PointCloud<pcl::PointXYZ>::Ptr tempPressHumid (new pcl::PointCloud<pcl::PointXYZ>);
	loadCloud("weather.txt", *tempPressHumid) ;


	//input of normal scale radius 
	double RAD; 
	double NN; 
	double Cal; 
	double tStep; 
	double maxdist;
	double factor;
	int K=10;
	int minPoints;

	//QFind reference cloud in file system, point clouds must have .xyz file extensions 
	QDir dir(".");
	QStringList filters;
	filters << "*.xyz"; 
	QFileInfoList list = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
	QFileInfo fileInfo(list[0].fileName());
	QString filenameref(fileInfo.fileName());

	int numFiles=list.count(); 

	// 7 command line arguements 
	RAD =atof(argv[1]);// Normal Calculation Radius 
	NN =atof(argv[2]);// Neighbourhood Radius for spatial averaging 
	Cal=atof(argv[3]); // Number of Calibration Clouds 
	tStep=atof(argv[4]); // Number of Time Clouds 
	maxdist=atof(argv[5]); // Maximum distance between point clouds, if greater= NaN 
	factor=atof(argv[6]); // factor times mean point spacing to search for point in the data cloud during raw distance calculation 
	minPoints=atof(argv[7]);// minimum number of points required or else point cloud is rejected 

	Eigen::MatrixXf CloudProp(numFiles,2);

	PCDWriter w;
	loadCloud(filenameref.toStdString(), *cloud_ref); // Populate reference cloud 

	tempCorrectModel(cloud_ref, tempPressHumid, 0); // Ciddor Correction 

	FilterInitial (cloud_ref); // Filter Outliers 
	cloud_ref= boxClip (cloud_ref);// Remove points outside specified area, required text file with numberical bounds 

	cout << "loaded reference cloud" << endl;

	pcl::PointCloud<pcl::Normal>::Ptr cloud_normals= normalCalcFunction (cloud_ref, RAD); // Calculates reference normals 

	double resolutionref = computeCloudResolution(cloud_ref); // calculates mean resolution of reference point cloud 
	double factorSearch= NN; 

	Eigen::Vector4f centroid_ref;
	pcl::compute3DCentroid(*cloud_ref,centroid_ref); // computes 3D centroid of reference point cloud 
	cout << centroid_ref << endl;
	cout << "Normals Calculated" << endl;
	tt.toc();
	tt.toc_print();

	// Setup Kd tree with reference cloud for NN

	pcl::copyPointCloud(*cloud_ref,*cloud_ref_align);
	pcl::copyPointCloud(*cloud_ref,*cloud_ref_noI);

	pcl::KdTreeFLANN<pcl::PointXYZ> kdtree_ref (new pcl::KdTreeFLANN<pcl::PointXYZ>);
	kdtree_ref.setInputCloud (cloud_ref_align); // computes reference cloud Kdtree
	cout << "kd tree built" << endl;

	// Storage Matrices 
	Eigen::MatrixXf calMat(cloud_ref->points.size (),Cal);
	Eigen::MatrixXf filteredMat(cloud_ref->points.size (),tStep); // fills all filtered data 
	Eigen::MatrixXf CalVector(cloud_ref->points.size (),1); 
	Eigen::MatrixXf FiltstdVector(cloud_ref->points.size (),1); 
	Eigen::MatrixXf CalstdVector(cloud_ref->points.size (),1);
	Eigen::MatrixXf timeFiltered(cloud_ref->points.size (),1); // Matrix to hold filtered Data
	Eigen::MatrixXf conftimeFiltered(cloud_ref->points.size (),1);

	int calcount;
	int jj=0;


	while (jj<numFiles-1) // -1 because of reference scan and another -1 because starts at 0
	{
		jj++;
		tt.tic();

		// Calibration Phase 
		if (jj <= Cal )     

		{
			cout << "calibrating"<< endl;
			pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_data (new pcl::PointCloud<pcl::PointXYZI>); // data cloud 

			//Query current file folder 
			QDir dir(".");
			QStringList filters;
			filters << "*.xyz"; 
			QFileInfoList list = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
			QFileInfo fileInfo(list[jj].fileName());
			QString filename(fileInfo.fileName());
			cout << filename.toStdString() << endl;
			loadCloud(filename.toStdString(), *cloud_data) ;

			FilterInitial (cloud_data); // Outlier removal
			cloud_data= boxClip (cloud_data);// Passthrough filter
			CloudProp(jj,0)=cloud_data->points.size ();
			CloudProp(jj,1)=computeCloudResolution(cloud_data);

			// testing to see if upper half is missing (Quality Control Step)

			pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_test (new pcl::PointCloud<pcl::PointXYZI>);
			pcl::copyPointCloud(*cloud_data, *cloud_test);

			pcl::PassThrough<pcl::PointXYZI> passt;
			float yup=2000; // Specific to study case 
			float ylow=centroid_ref(1);

			passt.setInputCloud (cloud_test);
			passt.setFilterFieldName ("y");
			passt.setFilterLimits (ylow, yup);
			passt.filter (*cloud_test);

			cout << "ylow" << endl; 
			cout << ylow << endl; 
			cout << cloud_test->points.size() << endl; 


			if  (cloud_data->points.size () <minPoints || cloud_test->points.size()==0)
			{
				cout << "skip scan not enough points" << endl; 

				if (jj==numFiles-1) // wait until more clouds are added 

				{

					int nf = numFiles;

					while(nf <= numFiles) {
						//cout << "waiting"<< endl;
						Sleep(10000);
						QDir dir(".");
						QStringList filters;
						filters << "*.xyz"; 
						QFileInfoList list2 = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
						nf=list2.count();
						cout << nf << endl;
					} 
					numFiles =nf; 
					Sleep(10000);
				}

				continue;
			}


			tempCorrectModel(cloud_data, tempPressHumid, jj);// Ciddor Correction
			pcl::copyPointCloud(*cloud_data, *cloud_data_align);
			pcl::PointCloud<pcl::Normal>::Ptr cloud_normalsdata= normalCalcFunctiondown (cloud_data_align, 20);
			roughAlign (cloud_ref_align,cloud_data_align);
			scannerTargetDistance(cloud_ref,cloud_data,jj); // 
			alignCloudsTran (cloud_data_align,cloud_ref_align);
			cout << "Transition Aligned"<< endl;

			alignCloudsFine (cloud_data_align,cloud_ref_align);

			w.writeBinaryCompressed ("Aligned_Corrected"+filename.section(".",0,0).toStdString()+".pcd", *cloud_data_align); // write the aligned point cloud to file 
			pcl::PointCloud<pcl::PointXYZI>::Ptr Rawdist= rawDistCalc (cloud_data_align,cloud_ref_noI,cloud_normals, K, maxdist,factor);
			cout << "Rawdist done"<< endl;
			cout << cloud_ref->size()<< endl;
			cout << Rawdist->size()<< endl;

			std::ofstream outputraw("Rawdiffs"+filename.section(".",0,0).toStdString()+".txt"); // write text file of raw differences 


			for(int ii=0; ii<cloud_ref->points.size ();ii++)
			{
				outputraw <<Rawdist->points[ii].intensity<< endl;  
			} 

			pcl::PointCloud<pcl::PointXYZI>::Ptr Filtdist=filtDistRad (factorSearch,resolutionref,Rawdist,kdtree_ref,cloud_ref_noI); //Spatial Filtering 
			cout << "Filtdist done"<< endl;

			w.writeBinaryCompressed ("Rawdsit"+filename.section(".",0,0).toStdString()+".pcd", *Rawdist); //Write raw distance point cloud to file 
			w.writeBinaryCompressed ("filteredspace"+filename.section(".",0,0).toStdString()+".pcd", *Filtdist);//Write spatial filter point cloud to file 

			// Add filtered distances to calibration matrix
			for(int ii=0; ii<cloud_ref->points.size ();ii++)

			{
				calMat.col(jj-1)[ii] = Filtdist->points[ii].intensity ;  	
			}


			if (jj == Cal )// once calMat matrix is built need to build calibration vector consisting of the average of each row 
			{
				cout << "last cal" << endl; 

				for(int ii=0; ii<cloud_ref->points.size ();ii++) // 
				{ 	
					float sum = 0;
					int den=0; 
					for (int j=0; j<Cal;j++) // go through each column sort out NaNs 
					{
						if (calMat.row(ii)[j]!=calMat.row(ii)[j])
						{
							// do nothing if NaN 
						}
						else
						{
							// if it is a number append to NonNan vector 
							den=den+1; 
							sum+=calMat.row(ii)[j]; 
						}
					}

					if (den==Cal)
					{
						calcount=den;

						std::vector<float> stdacc(den);
						for(int j=0;j<den;j++)
						{
							stdacc[j]=calMat.row(ii)[j];
						}
						double mean; 
						double stdcal;
						pcl::getMeanStd(stdacc,mean,stdcal); 
						CalstdVector(ii)=stdcal; 
						CalVector(ii)=mean;
					}

					else {

						CalVector(ii)=std::numeric_limits<double>::quiet_NaN();
						CalstdVector(ii)=std::numeric_limits<double>::quiet_NaN();
						calcount=0; 

					}
				}

			}
			tt.toc();
			tt.toc_print();

			if (jj==numFiles-1) {

				int nf = numFiles;

				while(nf <= numFiles) {
					Sleep(10000);
					QDir dir(".");
					QStringList filters;
					filters << "*.xyz"; // need to sort by time or number 
					QFileInfoList list2 = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
					nf=list2.count();
					cout << nf << endl;

				} 
				numFiles =nf; 
				Sleep(10000);
			}
			continue;
		}


		// Start Accumulation Phase 
		if (jj < Cal+tStep )    
		{

			cout << "Accumulating"<< endl;
			pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_data (new pcl::PointCloud<pcl::PointXYZI>);

			QDir dir(".");
			QStringList filters;
			filters << "*.xyz"; 
			QFileInfoList list = dir.entryInfoList(filters, QDir::Files); 
			QFileInfo fileInfo(list[jj].fileName());
			QString filename(fileInfo.fileName());
			cout << filename.toStdString() << endl;
			loadCloud(filename.toStdString(), *cloud_data) ;
			FilterInitial (cloud_data); 
			cloud_data= boxClip (cloud_data);

			CloudProp(jj,0)=cloud_data->points.size ();
			CloudProp(jj,1)=computeCloudResolution(cloud_data);

			// testing to see if upper half is missing (Quality Control Step)

			pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_test (new pcl::PointCloud<pcl::PointXYZI>);
			pcl::copyPointCloud(*cloud_data, *cloud_test);

			pcl::PassThrough<pcl::PointXYZI> passt;
			float yup=2000; // Specifil to study case
			float ylow=centroid_ref(1);

			passt.setInputCloud (cloud_test);
			passt.setFilterFieldName ("y");
			passt.setFilterLimits (ylow, yup);
			passt.filter (*cloud_test);

			cout << "ylow" << endl; 
			cout << ylow << endl; 

			if  (cloud_data->points.size () <minPoints || cloud_test->points.size()==0)
			{
				cout << "skip scan not enough points" << endl; 

				if (jj==numFiles-1)

				{

					int nf = numFiles;

					while(nf <= numFiles) {
						//cout << "waiting"<< endl;
						Sleep(10000);
						QDir dir(".");
						QStringList filters;
						filters << "*.xyz"; 
						QFileInfoList list2 = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
						nf=list2.count();
						cout << nf << endl;

					} 
					numFiles =nf; 
					Sleep(10000);
				}

				continue;
			}


			tempCorrectModel(cloud_data, tempPressHumid, jj); //Ciddor Correction
			pcl::copyPointCloud(*cloud_data, *cloud_data_align);
			roughAlign (cloud_ref_align,cloud_data_align);
			scannerTargetDistance(cloud_ref,cloud_data,jj);
			alignCloudsTran (cloud_data_align,cloud_ref_align);
			cout << "Transition Aligned"<< endl;
			alignCloudsFine (cloud_data_align,cloud_ref_align);
			cout << "Finely Aligned"<< endl;

			pcl::PointCloud<pcl::PointXYZI>::Ptr Rawdist= rawDistCalc (cloud_data_align,cloud_ref_noI,cloud_normals, K, maxdist,factor);
			cout << "Rawdist done"<< endl;
			cout << cloud_ref->size()<< endl;
			cout << Rawdist->size()<< endl;

			std::ofstream outputraw("Rawdiffs"+filename.section(".",0,0).toStdString()+".txt");

			for(int ii=0; ii<cloud_ref->points.size ();ii++)
			{
				outputraw <<Rawdist->points[ii].intensity<< endl;  
			} 

			pcl::PointCloud<pcl::PointXYZI>::Ptr Filtdist=filtDistRad (factorSearch,resolutionref,Rawdist,kdtree_ref,cloud_ref_noI);
			cout << "Filtdist done"<< endl;

			w.writeBinaryCompressed ("Rawdsit"+filename.section(".",0,0).toStdString()+".pcd", *Rawdist);
			w.writeBinaryCompressed ("filteredspace"+filename.section(".",0,0).toStdString()+".pcd", *Filtdist);
			w.writeBinaryCompressed ("Aligned_Corrected"+filename.section(".",0,0).toStdString()+".pcd", *cloud_data_align);

			// populate first timestep amount in matrix

			for(int ii=0; ii<cloud_ref->points.size ();ii++)
			{	
				filteredMat.col(jj-1-Cal)[ii]= Filtdist->points[ii].intensity;  
			}

			tt.toc();
			tt.toc_print();

			if (jj==numFiles-1) {
				int nf = numFiles; 
				while(nf <= numFiles) {
					//cout << "waiting"<< endl;
					Sleep(10000);
					QDir dir(".");
					QStringList filters;
					filters << "*.xyz"; 
					QFileInfoList list2 = dir.entryInfoList(filters, QDir::Files); 
					nf=list2.count();
					cout << nf << endl;

				} 
				numFiles =nf; 
				Sleep(10000);
			}
			continue;
		}

		// Start Time Filtering 
		if (jj >= Cal+tStep )  {

			// Time based Filtering 
			cout << "Filtering"<< endl;

			pcl::PointCloud<pcl::PointXYZI>::Ptr Filtconf (new pcl::PointCloud<pcl::PointXYZI>);
			Filtconf->width = cloud_ref->points.size ();
			Filtconf->height = 1;
			Filtconf->points.resize (Filtconf->width * Filtconf->height);

			pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_data (new pcl::PointCloud<pcl::PointXYZI>);
			QDir dir(".");
			QStringList filters;
			filters << "*.xyz";  
			QFileInfoList list = dir.entryInfoList(filters, QDir::Files); 

			QFileInfo fileInfo(list[jj].fileName());
			QString filename(fileInfo.fileName());
			cout << filename.toStdString() << endl;

			loadCloud(filename.toStdString(), *cloud_data) ;

			FilterInitial (cloud_data);
			cloud_data= boxClip (cloud_data);
			CloudProp(jj,0)=cloud_data->points.size ();
			CloudProp(jj,1)=computeCloudResolution(cloud_data);

			// testing to see if upper half is missing (Quality Control Step)

			pcl::PointCloud<pcl::PointXYZI>::Ptr cloud_test (new pcl::PointCloud<pcl::PointXYZI>);
			pcl::copyPointCloud(*cloud_data, *cloud_test);
			pcl::PassThrough<pcl::PointXYZI> passt;
			float yup=2000; 
			float ylow=centroid_ref(1);

			passt.setInputCloud (cloud_test);
			passt.setFilterFieldName ("y");
			passt.setFilterLimits (ylow, yup);
			passt.filter (*cloud_test);

			cout << "ylow" << endl; 
			cout << ylow << endl; 

			if  (cloud_data->points.size () <minPoints || cloud_test->points.size()==0)
			{
				cout << "skip scan not enough points" << endl; 

				if (jj==numFiles-1)

				{

					int nf = numFiles;

					while(nf <= numFiles) {
						//cout << "waiting"<< endl;
						Sleep(10000);
						QDir dir(".");
						QStringList filters;
						filters << "*.xyz"; // need to sort by time or number 
						QFileInfoList list2 = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
						nf=list2.count();
						cout << nf << endl;

					} 
					numFiles =nf; 
					Sleep(10000);
				}

				continue;
			}

			tempCorrectModel(cloud_data, tempPressHumid, jj);
			pcl::copyPointCloud(*cloud_data, *cloud_data_align);
			roughAlign (cloud_ref_align,cloud_data_align);
			scannerTargetDistance(cloud_ref,cloud_data,jj); // target based correction specific to study case 
			alignCloudsTran (cloud_data_align,cloud_ref_align);
			cout << "Transition Aligned"<< endl;
			alignCloudsFine (cloud_data_align,cloud_ref_align);
			cout << "Finely Aligned"<< endl;

			pcl::PointCloud<pcl::PointXYZI>::Ptr Rawdist= rawDistCalc (cloud_data_align,cloud_ref_noI,cloud_normals, K, maxdist,factor);
			cout << "Rawdist done"<< endl;
			cout << cloud_ref->size()<< endl;
			cout << Rawdist->size()<< endl;

			// Write text file of differences

			std::ofstream outputraw("Rawdiffs"+filename.section(".",0,0).toStdString()+".txt");

			for(int ii=0; ii<cloud_ref->points.size ();ii++)
			{
				outputraw <<Rawdist->points[ii].intensity<< endl;  
			} 

			pcl::PointCloud<pcl::PointXYZI>::Ptr Filtdist=filtDistRad (factorSearch,resolutionref,Rawdist,kdtree_ref,cloud_ref_noI);

			cout << "Filtdist done"<< endl;

			w.writeBinaryCompressed ("Aligned_Corrected"+filename.section(".",0,0).toStdString()+".pcd", *cloud_data_align);
			w.writeBinaryCompressed ("Rawdsit"+filename.section(".",0,0).toStdString()+".pcd", *Rawdist);
			w.writeBinaryCompressed ("filteredspace"+filename.section(".",0,0).toStdString()+".pcd", *Filtdist);

			if (tStep>0 )
			{
				for(int ii=0; ii<cloud_ref->points.size ();ii++)  
				{
					filteredMat.col(tStep-1)[ii]= Filtdist->points[ii].intensity;  	
					float sum = 0;
					int den=0; 

					for (int j=0; j<tStep;j++) // go through each column sort out NaNs 
					{
						if (filteredMat.row(ii)[j]!=filteredMat.row(ii)[j])
						{
							// do nothing if NaN 
						}
						else
						{
							// if it is a number append to NonNan vector 
							den=den+1; 
							sum+=filteredMat.row(ii)[j]; 
						}
					}

					if (den==tStep)
					{

						std::vector<float> stdacc(den);
						calcount=den;
						for(int j=0;j<den;j++)
						{
							stdacc[j]=filteredMat.row(ii)[j];
						}	

						double mean; 
						double standard; 
						pcl::getMeanStd(stdacc,mean,standard);
						FiltstdVector(ii)=standard;

						if (CalVector(ii)!=CalVector(ii))
						{
							timeFiltered(ii)=std::numeric_limits<double>::quiet_NaN();
							conftimeFiltered(ii)=std::numeric_limits<double>::quiet_NaN();
						}
						else
						{
							timeFiltered(ii)=mean-CalVector(ii);
							conftimeFiltered(ii)=1.96*(std::sqrtf((std::pow(CalstdVector(ii),2))/calcount+(std::pow(FiltstdVector(ii),2))/den));
						}
					}
					else
					{
						timeFiltered(ii)= std::numeric_limits<double>::quiet_NaN();
						conftimeFiltered(ii)=std::numeric_limits<double>::quiet_NaN();
						// if den is zero then the entire row is NaN so assign NaN
					}
				}

				//estimate registration error/empirical error term 
				double reg; 

				std::vector<float> areasDist;
				std::vector<float> areas(4)  ;// four areas to check residuals 


				areas[0]=901295; // Specific to study case 
				areas[1]=2850488;				
				areas[2]=2762438;
				areas[3]=1465845;


				std::vector<int> pointId(30);
				std::vector<float> pointNKN(30);

				kdtree_ref.nearestKSearch (areas[0], 30, pointId, pointNKN);


				for (long jj = 0; jj <pointId.size (); ++jj)
				{
					if (timeFiltered(pointId[jj])!=timeFiltered(pointId[jj]))
					{

					}
					else
					{

						areasDist.push_back( timeFiltered(pointId[jj]));

					}

				} 

				kdtree_ref.nearestKSearch (areas[1], 30, pointId, pointNKN);

				for (long jj = 0; jj <pointId.size (); ++jj)
				{

					if (timeFiltered(pointId[jj])!=timeFiltered(pointId[jj]))
					{

					}
					else
					{

						areasDist.push_back( timeFiltered(pointId[jj]));

					}

				} 

				kdtree_ref.nearestKSearch (areas[2], 30, pointId, pointNKN);

				for (long jj = 0; jj <pointId.size (); ++jj)
				{

					if (timeFiltered(pointId[jj])!=timeFiltered(pointId[jj]))
					{

					}
					else
					{

						areasDist.push_back( timeFiltered(pointId[jj]));

					}

				} 
				kdtree_ref.nearestKSearch (areas[3], 30, pointId, pointNKN);

				for (long jj = 0; jj <pointId.size (); ++jj)
				{

					if (timeFiltered(pointId[jj])!=timeFiltered(pointId[jj]))
					{

					}
					else
					{

						areasDist.push_back( timeFiltered(pointId[jj]));

					}

				} 

				double meanreg; 
				double stdreg; 
				pcl::getMeanStd(areasDist,meanreg,stdreg);

				for(int ii=0; ii<cloud_ref->points.size ();ii++)
				{
					Filtdist->points[ii].intensity=timeFiltered(ii);
					Filtconf->points[ii].intensity=conftimeFiltered(ii)+2*stdreg; // add registration error 
					Filtconf->points[ii].x=Filtdist->points[ii].x;
					Filtconf->points[ii].y=Filtdist->points[ii].y;
					Filtconf->points[ii].z=Filtdist->points[ii].z;
				}
				// Shift block 1 column to the left 
				removeColumn(filteredMat,0); 

			}


			pcl::PointCloud<pcl::PointXYZINormal>::Ptr cloud_with_Normals (new pcl::PointCloud<pcl::PointXYZINormal>);
			pcl::copyPointCloud(*cloud_normals,*cloud_with_Normals);
			pcl::copyPointCloud(*Filtdist, *cloud_with_Normals);

			pcl::PointCloud<pcl::PointXYZINormal>::Ptr cloud_with_Normalsconf (new pcl::PointCloud<pcl::PointXYZINormal>);
			pcl::copyPointCloud(*cloud_normals, *cloud_with_Normalsconf);
			pcl::copyPointCloud(*Filtconf, *cloud_with_Normalsconf);

			// Write text file of differences 

			std::ofstream output("Differences"+filename.section(".",0,0).toStdString()+".txt");

			for(int ii=0; ii<cloud_ref->points.size ();ii++)
			{
				output <<Filtdist->points[ii].intensity<< endl;  
			} 

			std::ofstream output2("Confidence"+filename.section(".",0,0).toStdString()+".txt");

			for(int ii=0; ii<cloud_ref->points.size ();ii++)
			{
				output2 <<Filtconf->points[ii].intensity<< endl;  
			} 


			w.writeBinaryCompressed ("filtered"+filename.section(".",0,0).toStdString()+".pcd", *cloud_with_Normals);
			w.writeBinaryCompressed ("Confidence"+filename.section(".",0,0).toStdString()+".pcd", *cloud_with_Normalsconf);

			tt.toc();
			tt.toc_print();


			if (jj==numFiles-1) {
				int nf = numFiles; 
				while(nf <= numFiles) {
					//cout << "waiting"<< endl;
					Sleep(10000);
					QDir dir(".");
					QStringList filters;
					filters << "*.xyz"; // need to sort by time or number  
					QFileInfoList list2 = dir.entryInfoList(filters, QDir::Files); // list of all pcd files
					nf=list2.count();
					cout << nf << endl;

				} 
				numFiles =nf; 
				Sleep(10000);
			}

		}

	}
	// Write point number text file
	std::ofstream output("PointNumber.txt");
	for(int ii=0; ii<numFiles;ii++)
	{
		output <<CloudProp(ii,0)<< endl;  
	} 
	// Write resolution text file 
	std::ofstream output2("Resolution.txt");
	for(int ii=0; ii<numFiles;ii++)
	{
		output2 <<CloudProp(ii,1)<< endl;  
	} 
	cout << "done" << endl; 
	return (0);
}