%Numerical solution to the sediment and POC storage model described in 
%Torres et al. Esurf. 
%% Sediment transit time distribution parameters
gam = 1.2; % Pareto distribution parameter gamma
tmin = 1.2399; %minimum value for distribution
a = 0.8; % Pareto distribution parameter alpha
b = 120; % Pareto distribution parameter beta
p = 1E6; % number of particles
nt = 30; % max number of transport events 
xtrans = 100; %transport length scale (km)
%range of catchment areas (km2) predicted by xtrans and nt
Carea = (((1:1:nt).*xtrans)./1.4).^(1/0.5); 
tCut = 350; % Tcut value for dimensionalization
%% Random pareto generation %%%%%%%%%%%%
U = (rand(p,nt).^(-a-1)) +(tmin-1); % pareto generation
Xe = gam.*exprnd(b,p,nt); % exp generation
U(U>Xe) = NaN; %exp tempering
U_sum = cumsum(U,2); %cumulative sum of ages for n transport events
Uind = find(isfinite(U_sum(:,end))==1); %finding NaN values
ndTTD = U_sum(Uind,:); % removing NaN values
TTD = ndTTD.*tCut; %dimensionalizing the TTD
Stor = TTD(:,1); %Storage times for each transport event 
for m = 2:nt %re-calculating storage times due to NaN values
    Stor(:,m) = TTD(:,m)-TTD(:,m-1);
end
clear gam tmin a b U Xe U_sum Uind p tCut
%% OC cycling sub-model parameters %%%%%
lambda = 1.209E-4; %14C decay constant (probably shouldnt change)
RadMod = 1.176E-12; %14/12 of "modern" C
Csf = 1; %steady-state concentration of fast C (g/cm3)
Css = 0.8; %steady-state concentration of slow C (g/cm3)
kf = 0.01; %rate constant for fast C (yr^-1)
ks = 2E-5; %rate constant for slow C (yr^-1)
Pfrad = (Csf.*kf).*RadMod; % fast 14 production rate
Psrad = (Css.*ks).*RadMod; % slow 14 production rate
Cif = 0; %initial fast C concentration
Rif = 0; %initial fast 14C concentration
Cis = 0; %initial slow C concentration
Ris = 0; %initial slow 14C concentration
%% OC cycling sub-model solution
% Between transport events, the initial concentrations increase reflecting
% POC cycling during previous storage event and in-channel homogenezation
fastC = @(t) ((Cif-Csf) .* exp(-kf .* t)) + Csf; %Equations for first transport event
fastCrad = @(t) (Rif - (Pfrad/(kf+lambda))) .* exp(t.*(-kf-lambda)) + (Pfrad/(kf+lambda));
slowC = @(t) ((Cis-Css) .* exp(-ks .* t)) + Css;
slowCrad = @(t) (Ris - (Psrad/(ks+lambda))) .* exp(t.*(-ks-lambda)) + (Psrad/(ks+lambda));
%Change in OC concentrations for one storage event
OCf = fastC(Stor(:,1)); %concentration fast C
OCfR = fastCrad(Stor(:,1)); %concentration fast 14C
OCs = slowC(Stor(:,1)); %concentration slow C
OCsR = slowCrad(Stor(:,1)); %concentration slow 14C
for m = 2:nt % loop for each additional transport event
    Cif = mean(OCf(:,m-1)); % new inital concentrations for OC equations
    Rif = mean(OCfR(:,m-1));
    Cis = mean(OCs(:,m-1));
    Ris = mean(OCsR(:,m-1));
    %new OC equations
    slowC = @(t) ((Cis-Css) .* exp(-ks .* t)) + Css;
    slowCrad = @(t) (Ris - (Psrad/(ks+lambda))) .* exp(t.*(-ks-lambda)) + (Psrad/(ks+lambda));
    fastC = @(t) ((Cif-Csf) .* exp(-kf .* t)) + Csf;
    fastCrad = @(t) (Rif - (Pfrad/(kf+lambda))) .* exp(t.*(-kf-lambda)) + (Pfrad/(kf+lambda));
    %Change in OC concentrations for additional storage event
    OCf(:,m) = fastC(Stor(:,m)); 
    OCfR(:,m) = fastCrad(Stor(:,m));
    OCs(:,m) = slowC(Stor(:,m));
    OCsR(:,m) = slowCrad(Stor(:,m));
end
POCtt = OCf+OCs; %total POC concentrations
POCttR = OCfR+OCsR; %total PO14C concentrations
Frac = (POCtt+POCttR)./(sum(POCtt)+sum(POCttR)); %concentration weighting factor
Fmtt = (POCttR./POCtt)./RadMod; %total fraction modern
bulkFm = sum(Frac.*Fmtt); %concentration weighted Fm
 clear lambda RadMod Csf Css kf ks Pfrad Psrad fastC slowC fastCrad ...
     slowCrad OCtt OCttR Cif Cis m OCf OCfR OCs OCsR POCtt POCttq ...
     POCttqR Rif Ris Stor POCttR POCttRq 
%% Plot model results
figure(1)
%clf
yyaxis left
plot(Carea,mean(TTD),'k--','linewidth',1.5,'DisplayName','MTT')
ylabel('Mean Transit Time (years)')
hold on; yyaxis right
plot(Carea,bulkFm,'b-','DisplayName','Fm (concentration-weighted)')
ylabel('biospheric Fraction Modern')
xlabel('Catchment Area (km^2)')
legend('show')
set(gca,'fontsize',18)
axis tight
